import {
  Controller,
  Get,
  Query,
  Param,
  UseInterceptors,
  InternalServerErrorException,
} from '@nestjs/common';
import {
  LoggerInterceptor,
  PurchasedOffersAndOwnerTokensDto,
  PurchasesService,
} from '@api-shared';
import {
  ApiOperation,
  ApiTags,
  ApiResponse,
  ApiBearerAuth,
  ApiOkResponse,
  ApiQuery,
} from '@nestjs/swagger';

@ApiBearerAuth()
@UseInterceptors(LoggerInterceptor)
@Controller('purchases')
@ApiTags('Purchase - public API')
export class PurchasesController {
  constructor(private readonly purchasesService: PurchasesService) {}

  @Get('/offers')
  @ApiOperation({
    summary:
      'Given user address, returns offer IDs grouped by type. Can filter by a specific offer type.',
  })
  @ApiQuery({
    name: 'tokenType',
    enum: ['SUB', 'PAYG', 'PAYU'],
    required: false,
    description: 'Optional: Filter offers by a specific type (SUB, PAYG, PAYU)',
  })
  @ApiOkResponse({
    description:
      'Retrieves an object of offer IDs grouped by type. If tokenType query is provided, it returns only that type.',
    schema: {
      example: {
        SUB: ['oid1', 'oid2', 'oid3'],
      },
    },
  })
  @ApiResponse({
    status: 500,
    description: 'Failed to retrieve offer IDs for the user',
  })
  async getUserOfferIds(
    @Query() dto: PurchasedOffersAndOwnerTokensDto,
  ): Promise<Record<string, string[]>> {
    try {
      const offerIdsResult =
        await this.purchasesService.getUserDataAccessOfferIds(
          dto.userAddress,
          dto.tokenType,
        );
      return offerIdsResult;
    } catch (error) {
      console.error('Error fetching user offer IDs:', error);
      throw new InternalServerErrorException({
        success: false,
        message: 'Failed to retrieve user offer IDs.',
        error: error.message || error.toString(),
      });
    }
  }

  @Get('/data-access-tokens')
  @ApiOperation({
    summary:
      'Given a user address, returns token IDs grouped by type. Can filter by a specific token type.',
  })
  @ApiQuery({
    name: 'tokenType',
    enum: ['SUB', 'PAYG', 'PAYU'],
    required: false,
    description: 'Optional: Filter tokens by a specific type (SUB, PAYG, PAYU)',
  })
  @ApiOkResponse({
    description:
      'An object containing arrays of token IDs for each offer type. If tokenType query is provided, it returns only that type.',
    schema: {
      example: {
        SUB: ['123', '1234', '12345'],
      },
    },
  })
  @ApiResponse({
    status: 500,
    description: 'Failed to retrieve token IDs for the user',
  })
  async getUserTokenIds(
    @Query() dto: PurchasedOffersAndOwnerTokensDto,
  ): Promise<Record<string, string[]>> {
    try {
      const tokenIdsResult =
        await this.purchasesService.getUserDataAccessTokenIds(
          dto.userAddress,
          dto.tokenType,
        );
      return tokenIdsResult;
    } catch (error) {
      console.error('Error fetching user token IDs:', error);
      throw new InternalServerErrorException({
        success: false,
        message: 'Failed to retrieve user token IDs.',
        error: error.message || error.toString(),
      });
    }
  }
}
