import { ApiProperty, ApiPropertyOptional } from '@nestjs/swagger';
import { Transform } from 'class-transformer';

import {
  IsString,
  IsNumber,
  Matches,
  MaxLength,
  MinLength,
  Min,
  IsArray,
  ValidateIf,
  IsNumberString,
  Length,
  IsOptional,
  IsEnum,
} from 'class-validator';

export class TradingHistoryCommonRequestDto {
  @ApiPropertyOptional({
    description: 'Filter transactions by Offer IDs (comma-separated values).',
    required: false,
  })
  @IsOptional()
  @ValidateIf((_, value) => value !== undefined)
  @Transform(({ value }) => (Array.isArray(value) ? value : [value]))
  @IsArray()
  @IsString({ each: true })
  oid?: string[];

  @ApiPropertyOptional({
    description: 'Asset ids (aid) - optional',
    required: false,
  })
  @IsOptional()
  @ValidateIf((_, value) => value !== undefined)
  @Transform(({ value }) => (Array.isArray(value) ? value : [value]))
  @IsArray()
  @IsString({ each: true })
  assetId?: string[];
}

export type TradingHistoryResponseDto = [number, string, number][];

// Define Enum for trading statistics output
export enum StatsOutputEnumDto {
  ASSETS = 'assets',
  OFFERINGS = 'offerings',
  TRANSACTIONS = 'transactions',
}

// Define Enum for trading statistics output
export enum UnitsEnumDto {
  BLOCK = 'block',
  MINUTE = 'minute',
}

export class IntervalDTO {
  @ApiPropertyOptional({ description: 'Interval length (default: 5 blocks).' })
  @IsNumberString()
  @IsOptional()
  intervalLength: string;

  @ApiPropertyOptional({
    description: 'Interval endpoint (default: latest block or timestamp).',
  })
  @IsNumberString()
  @IsOptional()
  intervalEnd: string;

  @ApiProperty({
    enum: UnitsEnumDto,
    description: 'Interval unit (block or minute).',
  })
  @IsEnum(UnitsEnumDto, { message: 'Must be block or minute.' })
  intervalUnits: UnitsEnumDto = UnitsEnumDto.BLOCK; // Default: Block-based

  @ApiProperty({
    enum: StatsOutputEnumDto,
    description: 'Statistics type (assets, offerings, or transactions).',
  })
  @IsEnum(StatsOutputEnumDto, {
    message: 'Must be assets, offerings, or transactions.',
  })
  stats: StatsOutputEnumDto = StatsOutputEnumDto.ASSETS; // Default: Asset stats
}

export class Token1AddressDto {
  @ApiProperty()
  @IsString()
  @Matches('^0x[0-9a-fA-F]{40}$')
  @Length(42, 42)
  addr1: string;
}

export class Token2AddressDto {
  @ApiProperty()
  @IsString()
  @Matches('^0x[0-9a-fA-F]{40}$')
  @Length(42, 42)
  addr2: string;
}

export class Token1IndexDto {
  @ApiProperty()
  @Matches('^[0-9]+$')
  @Min(0)
  indexno1: string;
}

export class Token2IndexDto {
  @ApiProperty()
  @Matches('^[0-9]+$')
  @Min(0)
  indexno2: string;
}

export class TokenPairNoDto {
  @ApiProperty()
  @IsNumber()
  @MinLength(1)
  @MaxLength(65)
  pairno: string;
}
