import {
  Controller,
  Get,
  Param,
  InternalServerErrorException,
} from '@nestjs/common';
import { TradingHistoryService, GovernanceService } from '@api-shared';
import { ApiOperation, ApiParam, ApiTags } from '@nestjs/swagger';
import { BigNumber } from 'ethers';

//const logger = new Logger(TradingHistoryService.name);

@Controller('gov/v1.0/retrieve-trading-history')
@ApiTags('Trading History')
export class TradingHistoryController {
  constructor(
    private readonly tradingHistoryService: TradingHistoryService,
    private readonly governanceService: GovernanceService,
  ) {}

  @Get('getNumberOfTradePairs/')
  @ApiOperation({ summary: 'Get number of trading pairs' })
  async getNumberOfTradePairs(): Promise<any> {
    try {
      const retnum = await this.tradingHistoryService.getNumberOfTradePairs();
      return retnum;
    } catch (error) {
      throw new InternalServerErrorException(
        'Failed to retrieve number of token trading pairs',
      );
    }
  }

  @Get('getIthTradePair/:ithp')
  @ApiOperation({ summary: 'Get trading pair numbered i' })
  @ApiParam({ name: 'ithp', type: Number, description: 'ith trading pair' })
  //@UsePipes(new ValidationPipe({ transform: true }))
  async getIthTradePair(@Param('ithp') ithp: string): Promise<any> {
    try {
      const iithp = Number(ithp);
      const retvals = await this.tradingHistoryService.getIthTradePair(iithp);
      return retvals;
    } catch (error) {
      throw new InternalServerErrorException(
        'Failed to retrieve ith token trading pair',
      );
    }
  }

  @Get('getTotalNumVolBuys/:ta1/:ti1/:ta2/:ti2')
  @ApiOperation({
    summary:
      'Get total number/volume of  buy transactions for token trading pair ( [ta1,ti1] => [ta2,ti2] )',
  })
  @ApiParam({
    name: 'ta1',
    type: String,
    description: '1st token contract ethereum address in token pair',
  })
  @ApiParam({
    name: 'ti1',
    type: Number,
    description:
      '1st token index in token pair (note: enter 0 for ERC20 tokens)',
  })
  @ApiParam({
    name: 'ta2',
    type: String,
    description: '2nd token contract ethereum address in token pair',
  })
  @ApiParam({
    name: 'ti2',
    type: Number,
    description:
      '2nd token index in token pair (note: enter 0 for ERC20 tokens)',
  })
  async getTotalNumVolBuys(
    @Param('ta1') adto1: string /* Token1AddressDto */,
    @Param('ti1') idto1: string /* Token1IndexDto */,
    @Param('ta2') adto2: string /* Token2AddressDto */,
    @Param('ti2') idto2: string /* Token2IndexDto */,
  ): Promise<any> {
    try {
      const retvals = await this.tradingHistoryService.getTotalNumVolBuys(
        adto1,
        BigNumber.from(idto1),
        adto2,
        BigNumber.from(idto2),
      );
      return retvals;
    } catch (error) {
      throw new InternalServerErrorException(
        'Failed in getTotalNumVolBuys call.',
      );
    }
  }

  @Get('getTotalNumVolSells/:ta1/:ti1/:ta2/:ti2')
  @ApiOperation({
    summary:
      'Get total number/volume of sell transactions for token trading pair ( [ta1,ti1] => [ta2,ti2] )',
  })
  @ApiParam({
    name: 'ta1',
    type: String,
    description: '1st token contract ethereum address in token pair',
  })
  @ApiParam({
    name: 'ti1',
    type: Number,
    description:
      '1st token index in token pair (note: enter 0 for ERC20 tokens)',
  })
  @ApiParam({
    name: 'ta2',
    type: String,
    description: '2nd token contract ethereum address in token pair',
  })
  @ApiParam({
    name: 'ti2',
    type: Number,
    description:
      '2nd token index in token pair (note: enter 0 for ERC20 tokens)',
  })
  async getTotalNumVolSells(
    @Param('ta1') ta1: string,
    @Param('ti1') ti1: string,
    @Param('ta2') ta2: string,
    @Param('ti2') ti2: string,
  ): Promise<any> {
    try {
      const iti1 = BigNumber.from(ti1);
      const iti2 = BigNumber.from(ti2);
      const retvals = await this.tradingHistoryService.getTotalNumVolSells(
        ta1,
        iti1,
        ta2,
        iti2,
      );
      return retvals;
    } catch (error) {
      throw new InternalServerErrorException(
        'Failed in getTotalNumVolSells call.',
      );
    }
  }

  @Get('getNumberOfTraderBuys/:tid/:ta1/:ti1/:ta2/:ti2')
  @ApiOperation({
    summary:
      'Get number of trader buy transactions for token trading pair ( [ta1,ti1] => [ta2,ti2] )',
  })
  @ApiParam({
    name: 'tid',
    type: String,
    description: 'Trader id which is an ethereum address',
  })
  @ApiParam({
    name: 'ta1',
    type: String,
    description: '1st token contract ethereum address in token pair',
  })
  @ApiParam({
    name: 'ti1',
    type: Number,
    description:
      '1st token index in token pair (note: enter 0 for ERC20 tokens)',
  })
  @ApiParam({
    name: 'ta2',
    type: String,
    description: '2nd token contract ethereum address in token pair',
  })
  @ApiParam({
    name: 'ti2',
    type: Number,
    description:
      '2nd token index in token pair (note: enter 0 for ERC20 tokens)',
  })
  async getNumberOfTraderBuys(
    @Param('tid') tid: string,
    @Param('ta1') ta1: string,
    @Param('ti1') ti1: string,
    @Param('ta2') ta2: string,
    @Param('ti2') ti2: string,
  ): Promise<any> {
    try {
      const iti1 = BigNumber.from(ti1);
      const iti2 = BigNumber.from(ti2);
      const retvals = await this.tradingHistoryService.getNumberOfTraderBuys(
        tid,
        ta1,
        iti1,
        ta2,
        iti2,
      );
      return retvals;
    } catch (error) {
      throw new InternalServerErrorException(
        'Failed in getNumberOfTraderBuys call.',
      );
    }
  }

  @Get('getNumberOfTraderSells/:tid/:ta1/:ti1/:ta2/:ti2')
  @ApiOperation({
    summary:
      'Get number of trader sell transactions for token trading pair ( [ta1,ti1] => [ta2,ti2] )',
  })
  @ApiParam({
    name: 'tid',
    type: String,
    description: 'Trader id which is an ethereum address',
  })
  @ApiParam({
    name: 'ta1',
    type: String,
    description: '1st token contract ethereum address in token pair',
  })
  @ApiParam({
    name: 'ti1',
    type: Number,
    description:
      '1st token index in token pair (note: enter 0 for ERC20 tokens)',
  })
  @ApiParam({
    name: 'ta2',
    type: String,
    description: '2nd token contract ethereum address in token pair',
  })
  @ApiParam({
    name: 'ti2',
    type: Number,
    description:
      '2nd token index in token pair (note: enter 0 for ERC20 tokens)',
  })
  async getNumberOfTraderSells(
    @Param('tid') tid: string,
    @Param('ta1') ta1: string,
    @Param('ti1') ti1: string,
    @Param('ta2') ta2: string,
    @Param('ti2') ti2: string,
  ): Promise<any> {
    try {
      const iti1 = BigNumber.from(ti1);
      const iti2 = BigNumber.from(ti2);
      const retvals = await this.tradingHistoryService.getNumberOfTraderSells(
        tid,
        ta1,
        iti1,
        ta2,
        iti2,
      );
      return retvals;
    } catch (error) {
      throw new InternalServerErrorException(
        'Failed in getNumberOfTraderSells call.',
      );
    }
  }

  @Get('getIthTraderSell/:ith/:tid/:ta1/:ti1/:ta2/:ti2')
  @ApiOperation({
    summary:
      'Get ith trader sell transaction for token trading pair ( [ta1,ti1] => [ta2,ti2] )',
  })
  @ApiParam({
    name: 'ith',
    type: Number,
    description: 'ith sell transaction : i=1,2,...',
  })
  @ApiParam({
    name: 'tid',
    type: String,
    description: 'Trader id which is an ethereum address',
  })
  @ApiParam({
    name: 'ta1',
    type: String,
    description: '1st token contract ethereum address in token pair',
  })
  @ApiParam({
    name: 'ti1',
    type: Number,
    description:
      '1st token index in token pair (note: enter 0 for ERC20 tokens)',
  })
  @ApiParam({
    name: 'ta2',
    type: String,
    description: '2nd token contract ethereum address in token pair',
  })
  @ApiParam({
    name: 'ti2',
    type: Number,
    description:
      '2nd token index in token pair (note: enter 0 for ERC20 tokens)',
  })
  async getIthTraderSell(
    @Param('ith') ith: string,
    @Param('tid') tid: string,
    @Param('ta1') ta1: string,
    @Param('ti1') ti1: string,
    @Param('ta2') ta2: string,
    @Param('ti2') ti2: string,
  ): Promise<any> {
    try {
      const iith = Number(ith);
      const iti1 = BigNumber.from(ti1);
      const iti2 = BigNumber.from(ti2);
      const retvals = await this.tradingHistoryService.getIthTraderSell(
        iith,
        tid,
        ta1,
        iti1,
        ta2,
        iti2,
      );
      return retvals;
    } catch (error) {
      throw new InternalServerErrorException(
        'Failed in getIthTraderSell call.',
      );
    }
  }

  @Get('getIthTraderBuy/:ith/:tid/:ta1/:ti1/:ta2/:ti2')
  @ApiOperation({
    summary:
      'Get ith trader buy transaction for token trading pair ( [ta1,ti1] => [ta2,ti2] )',
  })
  @ApiParam({
    name: 'ith',
    type: Number,
    description: 'ith buy transaction : i=1,2,...',
  })
  @ApiParam({
    name: 'tid',
    type: String,
    description: 'Trader id which is an ethereum address',
  })
  @ApiParam({
    name: 'ta1',
    type: String,
    description: '1st token contract ethereum address in token pair',
  })
  @ApiParam({
    name: 'ti1',
    type: Number,
    description:
      '1st token index in token pair (note: enter 0 for ERC20 tokens)',
  })
  @ApiParam({
    name: 'ta2',
    type: String,
    description: '2nd token contract ethereum address in token pair',
  })
  @ApiParam({
    name: 'ti2',
    type: Number,
    description:
      '2nd token index in token pair (note: enter 0 for ERC20 tokens)',
  })
  async getIthTraderBuy(
    @Param('ith') ith: string,
    @Param('tid') tid: string,
    @Param('ta1') ta1: string,
    @Param('ti1') ti1: string,
    @Param('ta2') ta2: string,
    @Param('ti2') ti2: string,
  ): Promise<any> {
    try {
      const iith = Number(ith);
      const iti1 = BigNumber.from(ti1);
      const iti2 = BigNumber.from(ti2);
      const retvals = await this.tradingHistoryService.getIthTraderBuy(
        iith,
        tid,
        ta1,
        iti1,
        ta2,
        iti2,
      );
      return retvals;
    } catch (error) {
      throw new InternalServerErrorException('Failed in getIthTraderBuy call.');
    }
  }

  @Get('getNumberOfSells/:ta1/:ti1/:ta2/:ti2')
  @ApiOperation({
    summary:
      'Get number of sell transactions for token trading pair ( [ta1,ti1] => [ta2,ti2] )',
  })
  @ApiParam({
    name: 'ta1',
    type: String,
    description: '1st token contract ethereum address in token pair',
  })
  @ApiParam({
    name: 'ti1',
    type: Number,
    description:
      '1st token index in token pair (note: enter 0 for ERC20 tokens)',
  })
  @ApiParam({
    name: 'ta2',
    type: String,
    description: '2nd token contract ethereum address in token pair',
  })
  @ApiParam({
    name: 'ti2',
    type: Number,
    description:
      '2nd token index in token pair (note: enter 0 for ERC20 tokens)',
  })
  async getNumberOfSells(
    @Param('ta1') ta1: string,
    @Param('ti1') ti1: string,
    @Param('ta2') ta2: string,
    @Param('ti2') ti2: string,
  ): Promise<any> {
    try {
      const iti1 = BigNumber.from(ti1);
      const iti2 = BigNumber.from(ti2);
      const retvals = await this.tradingHistoryService.getNumberOfSells(
        ta1,
        BigNumber.from(iti1),
        ta2,
        BigNumber.from(iti2),
      );
      return retvals;
    } catch (error) {
      throw new InternalServerErrorException(
        'Failed in getNumberOfSells call.',
      );
    }
  }

  @Get('getNumberOfBuys/:ta1/:ti1/:ta2/:ti2')
  @ApiOperation({
    summary:
      'Get number of buy transactions  for token trading pair ( [ta1,ti1] => [ta2,ti2] )',
  })
  @ApiParam({
    name: 'ta1',
    type: String,
    description: '1st token contract ethereum address in token pair',
  })
  @ApiParam({
    name: 'ti1',
    type: Number,
    description:
      '1st token index in token pair (note: enter 0 for ERC20 tokens)',
  })
  @ApiParam({
    name: 'ta2',
    type: String,
    description: '2nd token contract ethereum address in token pair',
  })
  @ApiParam({
    name: 'ti2',
    type: Number,
    description:
      '2nd token index in token pair (note: enter 0 for ERC20 tokens)',
  })
  async getNumberOfBuys(
    @Param('ta1') ta1: string,
    @Param('ti1') ti1: string,
    @Param('ta2') ta2: string,
    @Param('ti2') ti2: string,
  ): Promise<any> {
    try {
      const iti1 = BigNumber.from(ti1);
      const iti2 = BigNumber.from(ti2);
      const retvals = await this.tradingHistoryService.getNumberOfBuys(
        ta1,
        iti1,
        ta2,
        iti2,
      );
      return retvals;
    } catch (error) {
      throw new InternalServerErrorException('Failed in getNumberOfBuys call.');
    }
  }

  @Get('getIthBuy/:ith/:ta1/:ti1/:ta2/:ti2')
  @ApiOperation({
    summary:
      'Get ith trader buy transaction for token trading pair ( [ta1,ti1] => [ta2,ti2] )',
  })
  @ApiParam({
    name: 'ith',
    type: Number,
    description: 'ith buy transaction : i=1,2,...',
  })
  @ApiParam({
    name: 'ta1',
    type: String,
    description: '1st token contract ethereum address in token pair',
  })
  @ApiParam({
    name: 'ti1',
    type: Number,
    description:
      '1st token index in token pair (note: enter 0 for ERC20 tokens)',
  })
  @ApiParam({
    name: 'ta2',
    type: String,
    description: '2nd token contract ethereum address in token pair',
  })
  @ApiParam({
    name: 'ti2',
    type: Number,
    description:
      '2nd token index in token pair (note: enter 0 for ERC20 tokens)',
  })
  async getIthBuy(
    @Param('ith') ith: string,
    @Param('ta1') ta1: string,
    @Param('ti1') ti1: string,
    @Param('ta2') ta2: string,
    @Param('ti2') ti2: string,
  ): Promise<any> {
    try {
      const iith = Number(ith);
      const iti1 = BigNumber.from(ti1);
      const iti2 = BigNumber.from(ti2);
      const retvals = await this.tradingHistoryService.getIthBuy(
        iith,
        ta1,
        iti1,
        ta2,
        iti2,
      );
      return retvals;
    } catch (error) {
      throw new InternalServerErrorException('Failed in getIthBuy call.');
    }
  }

  @Get('getIthSell/:ith/:ta1/:ti1/:ta2/:ti2')
  @ApiOperation({
    summary:
      'Get ith trader sell transaction for token trading pair ( [ta1,ti1] => [ta2,ti2] )',
  })
  @ApiParam({
    name: 'ith',
    type: Number,
    description: 'ith sell transaction : i=1,2,...',
  })
  @ApiParam({
    name: 'ta1',
    type: String,
    description: '1st token contract ethereum address in token pair',
  })
  @ApiParam({
    name: 'ti1',
    type: Number,
    description:
      '1st token index in token pair (note: enter 0 for ERC20 tokens)',
  })
  @ApiParam({
    name: 'ta2',
    type: String,
    description: '2nd token contract ethereum address in token pair',
  })
  @ApiParam({
    name: 'ti2',
    type: Number,
    description:
      '2nd token index in token pair (note: enter 0 for ERC20 tokens)',
  })
  async getIthSell(
    @Param('ith') ith: string,
    @Param('ta1') ta1: string,
    @Param('ti1') ti1: string,
    @Param('ta2') ta2: string,
    @Param('ti2') ti2: string,
  ): Promise<any> {
    try {
      const iith = Number(ith);
      const iti1 = BigNumber.from(ti1);
      const iti2 = BigNumber.from(ti2);
      const retvals = await this.tradingHistoryService.getIthSell(
        iith,
        ta1,
        iti1,
        ta2,
        iti2,
      );
      return retvals;
    } catch (error) {
      throw new InternalServerErrorException('Failed in getIthSell call.');
    }
  }

  @Get('getBulkSells/:begrange/:endrange/:ta1/:ti1/:ta2/:ti2')
  @ApiOperation({
    summary:
      'Get bulk trader sell transactions in the range [begrange:endrange] for the token trading pair ( [ta1,ti1] => [ta2,ti2] )',
  })
  @ApiParam({
    name: 'begrange',
    type: Number,
    description: 'Beginning of range',
  })
  @ApiParam({ name: 'endrange', type: Number, description: 'End of range' })
  @ApiParam({
    name: 'ta1',
    type: String,
    description: '1st token contract ethereum address in token pair',
  })
  @ApiParam({
    name: 'ti1',
    type: Number,
    description:
      '1st token index in token pair (note: enter 0 for ERC20 tokens)',
  })
  @ApiParam({
    name: 'ta2',
    type: String,
    description: '2nd token contract ethereum address in token pair',
  })
  @ApiParam({
    name: 'ti2',
    type: Number,
    description:
      '2nd token index in token pair (note: enter 0 for ERC20 tokens)',
  })
  async getBulkSells(
    @Param('begrange') begrange: string,
    @Param('endrange') endrange: string,
    @Param('ta1') ta1: string,
    @Param('ti1') ti1: string,
    @Param('ta2') ta2: string,
    @Param('ti2') ti2: string,
  ): Promise<any> {
    try {
      const ibeg = Number(begrange);
      const iend = Number(endrange);
      const iti1 = BigNumber.from(ti1);
      const iti2 = BigNumber.from(ti2);
      const numsells = await this.tradingHistoryService.getNumberOfSells(
        ta1,
        iti1,
        ta2,
        iti2,
      );
      const allretvals = [];
      for (let i = ibeg; i < Math.min(iend, numsells); i++) {
        const retvals = await this.tradingHistoryService.getIthSell(
          i,
          ta1,
          BigNumber.from(iti1),
          ta2,
          BigNumber.from(iti2),
        );
        allretvals.push(retvals);
      }
      return allretvals;
    } catch (error) {
      throw new InternalServerErrorException('Failed in getBulkSells call.');
    }
  }

  @Get('getBulkBuys/:begrange/:endrange/:ta1/:ti1/:ta2/:ti2')
  @ApiOperation({
    summary:
      'Get bulk trader buy transactions in the range [begrange:endrange] for the token trading pair ( [ta1,ti1] => [ta2,ti2] )',
  })
  @ApiParam({
    name: 'begrange',
    type: Number,
    description: 'Beginning of range',
  })
  @ApiParam({ name: 'endrange', type: Number, description: 'End of range' })
  @ApiParam({
    name: 'ta1',
    type: String,
    description: '1st token contract ethereum address in token pair',
  })
  @ApiParam({
    name: 'ti1',
    type: Number,
    description:
      '1st token index in token pair (note: enter 0 for ERC20 tokens)',
  })
  @ApiParam({
    name: 'ta2',
    type: String,
    description: '2nd token contract ethereum address in token pair',
  })
  @ApiParam({
    name: 'ti2',
    type: Number,
    description:
      '2nd token index in token pair (note: enter 0 for ERC20 tokens)',
  })
  async getBulkBuys(
    @Param('begrange') begrange: string,
    @Param('endrange') endrange: string,
    @Param('ta1') ta1: string,
    @Param('ti1') ti1: string,
    @Param('ta2') ta2: string,
    @Param('ti2') ti2: string,
  ): Promise<any> {
    try {
      const ibeg = Number(begrange);
      const iend = Number(endrange);
      const iti1 = BigNumber.from(ti1);
      const iti2 = BigNumber.from(ti2);
      const numbuys = await this.tradingHistoryService.getNumberOfBuys(
        ta1,
        iti1,
        ta2,
        iti2,
      );
      const allretvals = [];
      for (let i = ibeg; i < Math.min(iend, numbuys); i++) {
        const retvals = await this.tradingHistoryService.getIthBuy(
          i,
          ta1,
          iti1,
          ta2,
          iti2,
        );
        allretvals.push(retvals);
      }
      return allretvals;
    } catch (error) {
      throw new InternalServerErrorException('Failed in getBulkBuys call.');
    }
  }
}
