import {
  Controller,
  Get,
  Query,
  Param,
  HttpException,
  HttpStatus,
  BadRequestException,
  InternalServerErrorException,
} from '@nestjs/common';
import { TradingHistoryService } from '@api-shared';
import {
  ApiBearerAuth,
  ApiOperation,
  ApiQuery,
  ApiResponse,
  ApiTags,
  ApiParam,
} from '@nestjs/swagger';
import { BigNumber } from 'ethers';

import {
  TradingHistoryCommonRequestDto,
  IntervalDTO,
  StatsOutputEnumDto,
  UnitsEnumDto,
} from './trading-history-dto/trading-history.dto';
@ApiBearerAuth()
@Controller('trading-history/')
@ApiTags('Trading History - public API')
export class TradingHistoryController {
  constructor(private readonly tradingHistoryService: TradingHistoryService) {}

  @Get('/')
  @ApiOperation({
    summary:
      'Retrieves historical sell transactions filtered by Offering ID (OID) or Asset ID.',
  })
  @ApiResponse({
    status: 200,
    description:
      'Returns an object where each key is an asset or OID, and the value is an array of sell transactions. Each transaction is represented as [price, label, timestamp].\n\n' +
      'Label is composed of: boursetx-{txid}-{contract-address-of-data-access}-{assetid}/{oid}-{payment-token-symbol}.',
    example: {
      sub1: [
        [
          10000000000000000000,
          'boursetx-1-0x9264AC53dCd1B4Ef3b295E7CeE0337384AF4B97a-sub1-FDE',
          1738083560,
        ],
      ],
      sub2: [
        [
          5000000000000000000,
          'boursetx-1-0x1234567890abcdef1234567890abcdef12345678-sub2-ABC',
          1738089999,
        ],
      ],
    },
    isArray: true,
  })
  @ApiQuery({
    name: 'oid',
    required: false,
    type: [String],
    isArray: true,
  })
  @ApiQuery({
    name: 'assetId',
    required: false,
    type: [String],
    isArray: true,
  })
  async getTradingHistoryCommon(
    @Query() oidDto: TradingHistoryCommonRequestDto,
  ): Promise<Record<string, any[]>> {
    const oids = oidDto.oid ? [].concat(oidDto.oid) : [];
    const assetIds = oidDto.assetId ? [].concat(oidDto.assetId) : [];

    if (oids.length === 0 && assetIds.length === 0) {
      throw new BadRequestException(
        `Please provide oids or assetIds in the request`,
      );
    }

    // Handle oids
    if (oids.length > 0) {
      const tradesArray = await Promise.all(
        oids.map((singleOid) =>
          this.tradingHistoryService.getAssetOrOIdTradeHistory(singleOid, true),
        ),
      );
      return oids.reduce(
        (acc, curr: string, index) => {
          acc[curr] = tradesArray[index];
          return acc;
        },
        {} as Record<string, any[]>,
      );
    }

    // Handle assetIds
    if (assetIds.length > 0) {
      const tradesArray = await Promise.all(
        assetIds.map((singleAssetId) =>
          this.tradingHistoryService.getAssetOrOIdTradeHistory(
            singleAssetId,
            false,
          ),
        ),
      );
      return assetIds.reduce(
        (acc, curr: string, index) => {
          acc[curr] = tradesArray[index];
          return acc;
        },
        {} as Record<string, any[]>,
      );
    }

    throw new BadRequestException(
      `Please provide valid oids or assetIds in the request`,
    );
  }

  @Get('/offering-sales-statistics')
  @ApiOperation({
    summary:
      'Fetches aggregated sales statistics for Offerings and Assets over a defined time or block range.',
  })
  @ApiResponse({
    status: 200,
    description:
      'Returns aggregated sales statistics based on the selected grouping:\n' +
      '- *Assets*: Stats per asset ID (AID) [assetid, min, avg, max price, stddev, payment token, amount, count].\n' +
      '- *Offerings*: Stats per offering ID (OID) [assetid, oid, price, payment token, amount, count].\n' +
      '- *Transactions*: Individual trades [assetid, oid, price, payment token, amount, timestamp].',
    example: [
      [
        ['accc2', 'occc21', '10.00', 'FDE', 3, 3],
        ['accc1', 'occc12', '12.00', 'FDE', 1, 1],
        ['accc1', 'occc13', '8.00', 'FDE', 1, 1],
        ['accc2', 'occc23', '13.00', 'FDE', 2, 2],
      ],
    ],
  })
  async getOfferingSalesStatistics(
    @Query() intervalDto: IntervalDTO,
  ): Promise<any> {
    const DEFAULT_INTERVALLENGTH = 5; // Default interval length
    let intervalLength: number;
    let intervalEnd: number;

    try {
      if (intervalDto.intervalLength === undefined) {
        intervalLength = DEFAULT_INTERVALLENGTH;
      } else {
        intervalLength = Number(intervalDto.intervalLength);
      }
      if (intervalDto.intervalUnits === UnitsEnumDto.BLOCK) {
        if (intervalDto.intervalEnd === undefined) {
          intervalEnd = await this.tradingHistoryService.getLatestBlockNo();
        } else {
          intervalEnd = Number(intervalDto.intervalEnd);
        }
      } else if (intervalDto.intervalUnits === UnitsEnumDto.MINUTE) {
        if (intervalDto.intervalEnd === undefined) {
          const currentTime = Math.floor(Date.now() / 1000); // get current time in unix time format
          intervalEnd = currentTime;
        } else {
          intervalEnd = Number(intervalDto.intervalEnd);
        }
        const latestBlockNo =
          await this.tradingHistoryService.getLatestBlockNo();
        const begTime = intervalEnd - intervalLength * 60;
        const begBlockNo = await this.tradingHistoryService.binarySearch(
          begTime,
          1,
          latestBlockNo,
        );
        const endBlockNo = await this.tradingHistoryService.binarySearch(
          intervalEnd,
          1,
          latestBlockNo,
        );
        //console.log("==1===>",begTime,intervalEnd,intervalLength,begBlockNo,endBlockNo) ;
        intervalEnd = endBlockNo;
        intervalLength = endBlockNo - begBlockNo;
        //console.log("==2==>",intervalLength,intervalEnd) ;
      }
      if (intervalDto.stats === StatsOutputEnumDto.ASSETS) {
        const salesStats =
          await this.tradingHistoryService.getAssetSalesStatistics(
            intervalLength,
            intervalEnd,
          );
        return salesStats;
      } else if (intervalDto.stats === StatsOutputEnumDto.OFFERINGS) {
        const salesStats =
          await this.tradingHistoryService.getOfferingSalesStatistics(
            intervalLength,
            intervalEnd,
          );
        return salesStats;
      } else {
        const b = await this.tradingHistoryService.getValidBlockRange(
          intervalLength,
          intervalEnd,
        );
        const salesStats =
          await this.tradingHistoryService.getSellRecordsInBlockRange(
            b.begBlockNo,
            b.endBlockNo,
          );
        return salesStats;
      }
    } catch (error) {
      throw new HttpException(
        'Getting offering (oid) sales statistics error',
        HttpStatus.NOT_FOUND,
      );
    }
  }

  @Get('trades-count')
  @ApiOperation({
    summary: 'Get total number of purchases',
    description:
      'Returns the total number of purchases across all trading pairs. A trading pair represents a data access token paired with a payment token (e.g., data access / FDE token).',
  })
  @ApiResponse({
    status: 200,
    description: 'Total number of purchases',
    schema: {
      type: 'number',
      example: 92,
    },
  })
  async getTradesCount(): Promise<number> {
    return await this.tradingHistoryService.getTotalPurchases();
  }
}
