import { INestApplication } from '@nestjs/common';
import { Test } from '@nestjs/testing';
import { AppModule as privateApp } from '../../../tm-private/src/app.module';
import { AppModule as publicApp } from '../../../tm-public/src/app.module';
import { ethers as hardhatEthers } from 'hardhat';
import '@openzeppelin/hardhat-upgrades';
import { ethers } from 'ethers';
import { InitializerModule } from '@api-shared';
import axios from 'axios'; // Add axios to send the RPC request

// eslint-disable-next-line @typescript-eslint/no-var-requires
require('dotenv').config({ path: `.env.test` });

import hartHatConfig from '../../../../hardhat.config';

let privateAppInstance: INestApplication;
let publicAppInstance: INestApplication;
let signerWithAddress;
let user1: ethers.Wallet | null = null;
let provider: ethers.providers.JsonRpcProvider;

export const offeringsArr = [
  {
    oid: 'sub1',
    resource: 'http://resource1.jpg',
    beneficiary: '0x57fFC30a883932F98fE94Fc0eE4390C4BA5A9F2a',
    price: '10',
    cap_duration: '31536000',
    cds_target: { key: 'value' },
    cds_sl: { key: 'value' },
  },
  {
    oid: 'sub2',
    resource: 'http://resource2.jpg',
    beneficiary: '0x57fFC30a883932F98fE94Fc0eE4390C4BA5A9F2a',
    price: '10',
    cap_duration: '31536000',
    cds_target: { key: 'value' },
    cds_sl: { key: 'value' },
  },
  {
    oid: 'sub3',
    resource: 'http://resource3.jpg',
    beneficiary: '0x57fFC30a883932F98fE94Fc0eE4390C4BA5A9F2a',
    price: '10',
    cap_duration: 31536000,
    cds_target: { key: 'value' },
    cds_sl: { key: 'value' },
  },
];

export const getSignerWallet = async (): Promise<any> => {
  if (!provider) {
    await getEthRpcProvider();
  }

  if (!user1) {
    user1 = new ethers.Wallet(
      hartHatConfig.networks.hardhat.accounts[1].privateKey,
      provider,
    );
  }

  return user1;
};
export const getEthRpcProvider = async (): Promise<any> => {
  if (!provider) {
    provider = new ethers.providers.JsonRpcProvider(`http://127.0.0.1:8545`);
  }

  return provider;
};
export const getPrivateApp = async (): Promise<INestApplication> => {
  if (!privateAppInstance) {
    const moduleRef = await Test.createTestingModule({
      imports: [privateApp],
    }).compile();

    privateAppInstance = moduleRef.createNestApplication();
    await privateAppInstance.init();
  }
  return privateAppInstance;
};
export const getPublicApp = async (): Promise<INestApplication> => {
  if (!publicAppInstance) {
    const moduleRef = await Test.createTestingModule({
      imports: [publicApp],
    }).compile();

    publicAppInstance = moduleRef.createNestApplication();
    await publicAppInstance.init();
  }
  return publicAppInstance;
};

export const getSigners = async () => {
  if (!signerWithAddress) {
    signerWithAddress = await hardhatEthers.getSigners();
  }

  return signerWithAddress;
};

export const closeApp = async () => {
  if (privateAppInstance) {
    await privateAppInstance.close();
  }
  if (publicAppInstance) {
    await publicAppInstance.close();
  }
};

//Time cap is defined by jest.testTimeout in project.json
export const waitForInitialization = async (module: InitializerModule) => {
  while (!module.isInitialized()) {
    await new Promise((resolve) => setTimeout(resolve, 100)); // Check every 100ms
  }
};

// Function to revert to a previous snapshot
export async function revertHardHatNodeToSnapshot(
  snapshotId = '0xc',
): Promise<void> {
  try {
    await axios.post('http://127.0.0.1:8545', {
      jsonrpc: '2.0',
      method: 'evm_revert',
      params: [snapshotId],
      id: 1,
    });
    console.log('Reverted to snapshot ID:', snapshotId);
  } catch (error) {
    console.error('Failed to revert to snapshot', error);
    throw error;
  }
}
