# TM UI Demo

Angular application demonstrating the FAME Trading & Monetisation (TM) system. This application serves as a proof-of-concept for testing the TM Public API and provides an interface for interacting with blockchain-based data marketplace features.

## Features

### Core Functionality
- Data Access Management: View and manage purchased data access tokens (SUB, PAYG, PAYU)
- MetaMask Integration: Connect and interact with MetaMask wallet
- Multi-Environment Support: Switch between local development and BESU network
- Token Validation: Validate data access permissions through smart contracts
- Purchase History: Track and display trading history

### User Interface
- Responsive Design: Built with Tailwind CSS
- Data Access Cards: Interactive cards displaying purchased offers with expandable details
- Real-time Loading States: Visual feedback during blockchain interactions
- Manual Asset Addition: Add custom data access assets for testing

## Tech Stack

- Framework: Angular 18
- Styling: Tailwind CSS
- Blockchain: viem for Ethereum interactions
- HTTP Client: Angular HttpClient for API communication
- State Management: Angular services with RxJS

## Prerequisites

Before running the application:

1. Node.js (v18 or higher)
2. npm package manager
3. MetaMask browser extension
4. TM Backend Services running:
   - TM Public API (default: `http://localhost:3001`)
   - Data Gateway Service (default: `http://localhost:7777`)
5. Blockchain Network:
   - Local: Hardhat node at `http://localhost:8545`
   - BESU: FAME BESU network at `http://162.55.94.15:8545`

## Setup and Execution

### Development Setup

1. Navigate to the app directory:
   ```bash
   cd ./tm/apps/tm-ui-demo
   ```

2. Install dependencies:
   ```bash
   npm install
   ```

3. Start the application:

   **Local Development (Hardhat)**:
   ```bash
   npm start
   ```
   - Uses `localhost:8545` as blockchain RPC
   - Connects to local TM APIs
   - Uses `ACCESS_DATA_MOCK_LOCAL` for demo data
   - Network ID: 31337

   **BESU Network**:
   ```bash
   npm run start:besu
   ```
   - Uses BESU network at `162.55.94.15:8545`
   - Connects to production APIs
   - Uses `ACCESS_DATA_MOCK` for demo data
   - Network ID: 6666

4. Access the application at `http://localhost:4200`

## Configuration

### Environment Files

- `src/environments/environment.ts`: Local development (Hardhat)
- `src/environments/environment.besu.ts`: BESU network configuration

### Configuration Structure

```typescript
export const environment = {
  production: false,
  api: {
    url: 'http://localhost:3001/api/v1.0',        // TM Public API
    dataAccessApiService: 'http://localhost:7777/api' // Data Gateway API
  },
  walletConfig: {
    id: 31337,              // Chain ID
    networkId: 31337,       // Network ID
    name: 'Hardhat Localhost',
    rpcUrls: {
      default: {
        http: ['http://localhost:8545']
      }
    }
  }
};
```

## Application Structure

### Routes
- `/` - Data Access dashboard showing purchased data access cards
- `/explorer` - Browse available data offerings
- `/trading-history` - View transaction history
- `/playground` - Development testing area

### Components

#### Data Access Cards (`src/components/data-access-cards/`)
- Displays purchased offers as interactive cards
- Fetches purchase data from blockchain via API
- Validates access tokens through smart contracts
- Supports manual addition of test assets

#### MetaMask Connect (`src/components/metamask-connect/`)
- Wallet connection and authentication
- Account switching and network detection
- Web3 interaction utilities

### Services

#### TM API Service (`src/services/tm-api.service.ts`)
- `fetchPurchasedOffers(userAddress)`: Get user's purchased offer IDs
- `fetchTokenIdForOffer(offerId)`: Get token ID for specific offer
- `fetchContractAddresses()`: Get deployed contract addresses
- `getApprovalTx()`: Get approval transaction for spending tokens
- `getSubmissionOrderTx(oid)`: Get submission order transaction

#### MetaMask Service (`src/services/metamask.service.ts`)
- Wallet connection management
- Transaction signing
- Network switching
- Account management

## Data Flow

1. User connects MetaMask wallet
2. Application fetches purchased offer IDs from TM API using wallet address
3. Application filters mock data based on purchased IDs and environment
4. User expands cards to load token IDs for validation
5. User validates access by signing messages and calling Data Gateway API

## Mock Data Configuration

The application uses hardcoded mock data to simulate offers published by the specific data provider whose service the user is accessing. The API returns purchased offer IDs, but the mock data represents only the catalog of offers that this particular provider has published, not all marketplace offers. This hybrid approach combines real blockchain purchase data with the provider's specific offer catalog for demonstration purposes.

### Mock Data Sets

#### `ACCESS_DATA_MOCK` (BESU Environment)
Located in `src/components/data-access-cards/data-access-data.ts`, contains production-like offer data:

```typescript
{
  offerId: '12CwhrbhP7wwWHkpxTRqthz8e8UZN6WXmrz1gk2oXVpaARtgJLA',
  tokenType: 'SUB',
  assetName: 'Daily Market Insights Subscription',
  assetDescription: 'Enjoy daily, expert financial analysis reports...'
}
```

- **Purpose**: Simulates offers published by this specific data provider with realistic IDs and descriptions
- **Token Types**: SUB (Subscription), PAYG (Pay-as-you-go), PAYU (Pay-per-use)
- **Offer IDs**: Long alphanumeric strings matching expected blockchain format
- **Use Case**: Testing against BESU network with production-like provider catalog

#### `ACCESS_DATA_MOCK_LOCAL` (Local Development)
Simplified data for local testing:

```typescript
{
  offerId: 'sub1',
  tokenType: 'SUB',
  assetName: 'Daily Market Insights Subscription',
  assetDescription: 'Enjoy daily, expert financial analysis reports...'
}
```

- **Purpose**: Easy-to-remember IDs for local development and testing
- **Offer IDs**: Simple strings ('sub1', 'sub2', 'payg1', 'payu1')
- **Use Case**: Quick testing without complex blockchain interactions

### Data Flow Implementation

1. **API Call**: `fetchPurchasedOffers(userAddress)` returns array of purchased offer IDs from blockchain
2. **Environment Detection**: RPC URL check determines which provider catalog (mock data set) to use
3. **Catalog Filtering**: Provider's offer catalog is filtered to show only offers that the user has purchased
4. **Display**: Filtered results show user's purchased offers from this specific provider's catalog

### Why Mock Data is Necessary

- **Provider-Specific Catalog**: Represents the specific offers published by this data provider, not the entire marketplace
- **API Limitation**: Backend only provides offer IDs, not complete offer metadata from provider's catalog
- **Demonstration Purpose**: Shows how users would see their purchased offers from this provider's service
- **Development Efficiency**: Allows frontend development without complete provider catalog management system
- **Testing Flexibility**: Enables testing different offer types and scenarios within a provider's scope

### Mock Data Structure

Each mock offer contains:
- `offerId`: Unique identifier matching blockchain/API format
- `tokenType`: 'SUB', 'PAYG', or 'PAYU'
- `assetName`: Display name for the data offering
- `assetDescription`: Detailed description of the offering
- `dataVolume`: (PAYG only) Data volume in KB for pay-per-consumption

Environment detection logic:
- **BESU**: RPC URL contains `162.55.94.15` → uses `ACCESS_DATA_MOCK`
- **Local**: Other RPC URLs → uses `ACCESS_DATA_MOCK_LOCAL`

### Future Implementation Notes

In a production system, the mock data would be replaced by:
- API endpoint returning provider's complete offer catalog: `GET /provider/offers`
- API endpoint for specific offer details: `GET /offers/{offerId}`
- Database-backed provider offer metadata management
- Dynamic offer creation and management interface for data providers

## Available Scripts

```bash
npm start          # Start with local configuration
npm run start:besu # Start with BESU configuration
npm run build      # Build for production
npm run watch      # Build with file watching
npm run test       # Run unit tests
```

## Testing Workflow

1. Ensure backend services are running
2. Connect MetaMask to appropriate network
3. Fund wallet with test tokens if needed
4. Purchase test offers through TM system
5. Verify offers appear in Data Access dashboard

## Troubleshooting

**MetaMask Connection Issues**:
- Verify MetaMask is installed and unlocked
- Check correct network selection
- Clear browser cache if needed

**No Offers Displayed**:
- Confirm wallet has purchased offers
- Verify backend APIs are running
- Check environment configuration

**API Communication Errors**:
- Confirm TM Public API is running on correct port
- Check Data Gateway Service availability
- Verify network connectivity to BESU (if using BESU environment)

## Related Documentation

- [FAME TM System Overview](../../CLAUDE.md)
- [TM Public API Documentation](../tm-public/)
- [Smart Contracts Documentation](../../libs/contracts/)
