import { Component, EventEmitter, OnInit, Output } from '@angular/core';
import { FormsModule, ReactiveFormsModule } from '@angular/forms';
import { MetaWalletService } from '../../services/metamask.service';
import { HttpClient } from '@angular/common/http';
import { CommonModule } from '@angular/common';

import { DataAccessCardComponent } from './data-access-card/data-access-card.component';
import {
  TmApiService,
  DataAccessContractAddresses,
} from '../../services/tm-api.service';
import { DataAccessServiceApiService } from '../../services/data-gateway-api.service';
import { AccessData, NewAccessCard, TokenType } from '../../types/types';
import { ACCESS_DATA_MOCK, ACCESS_DATA_MOCK_LOCAL } from './data-access-data';
import { environment } from '../../environments/environment';

@Component({
  selector: 'app-data-access-cards',
  standalone: true,
  imports: [
    ReactiveFormsModule,
    CommonModule,
    DataAccessCardComponent,
    FormsModule,
  ],
  providers: [MetaWalletService, TmApiService, HttpClient],
  templateUrl: './data-access-cards.component.html',
})
export class DataAccessCardsComponent implements OnInit {
  @Output() onResult = new EventEmitter<any>();

  accessCards: AccessData[] = [];
  result: any = null;
  constractAddresses: DataAccessContractAddresses | undefined;
  loading = false;
  menuOpen = false;
  showModal = false;

  newCard: NewAccessCard = {
    tokenType: 'SUB',
    offerId: '',
    assetName: '',
    assetDescription: '',
  };

  constructor(
    private metaWalletService: MetaWalletService,
    private tmApiService: TmApiService,
    private dataGatewayApiService: DataAccessServiceApiService,
  ) {}

  ngOnInit(): void {
    this.loadUserOffers();
    this.tmApiService
      .fetchContractAddresses()
      .then((res) => {
        this.constractAddresses = res;
      })
      .catch((error) => {
        console.error('Error fetching contract addresses:', error);
      });
  }

  async loadUserOffers() {
    this.loading = true;
    try {
      const userAddress = await this.metaWalletService.getAccount();
      const purchasedOffersObject =
        await this.tmApiService.fetchPurchasedOffers(userAddress);
      const purchasedOfferIds = new Set<string>();
      Object.values(purchasedOffersObject).forEach((idArray: any) => {
        idArray.forEach((id: string) => purchasedOfferIds.add(id));
      });
      const isBesuEnvironment =
        environment.walletConfig.rpcUrls.default.http[0].includes(
          '162.55.94.15',
        );
      const mockData = isBesuEnvironment
        ? ACCESS_DATA_MOCK
        : ACCESS_DATA_MOCK_LOCAL;
      const purchasedCards = mockData.filter((card: any) =>
        purchasedOfferIds.has(card.offerId),
      );

      const savedCardsRaw = localStorage.getItem('userAddedCards');
      const manuallyAddedCards: AccessData[] = savedCardsRaw
        ? JSON.parse(savedCardsRaw)
        : [];

      const finalCardsMap = new Map<string, AccessData>();

      purchasedCards.forEach((card: any) =>
        finalCardsMap.set(card.offerId, card),
      );

      manuallyAddedCards.forEach((card) =>
        finalCardsMap.set(card.offerId, card),
      );

      this.accessCards = Array.from(finalCardsMap.values());
    } catch (error) {
      console.error('Failed to load user offers:', error);
      const savedCardsRaw = localStorage.getItem('userAddedCards');
      this.accessCards = savedCardsRaw ? JSON.parse(savedCardsRaw) : [];
    } finally {
      this.loading = false;
    }
  }

  async handleCardExpansion(clickedCard: AccessData) {
    clickedCard.isExpanded = !clickedCard.isExpanded;

    if (clickedCard.isExpanded && !clickedCard.tokenId) {
      try {
        clickedCard.isLoadingToken = true;

        const response = await this.tmApiService.fetchTokenIdForOffer(
          clickedCard.offerId,
        );

        clickedCard.tokenId = response;
      } catch (error) {
        console.error(
          `Failed to fetch tokenId for ${clickedCard.offerId}`,
          error,
        );
      } finally {
        clickedCard.isLoadingToken = false;
      }
    }
  }

  async onSubmit(accessData: AccessData) {
    if (!accessData.tokenId) {
      this.result = {
        error: 'Token ID is missing. Cannot perform validation.',
      };
      return;
    }

    await this.performValidation(accessData);
  }

  async performValidation(accessData: AccessData) {
    const { tokenId, tokenType, dataVolume } = accessData;
    this.loading = true;
    try {
      console.log('Starting signing process...');
      const { address, message, signature } =
        await this.metaWalletService.signAccessMessage(
          tokenId!,
          tokenType as TokenType,
        );
      console.log('Signed!', { address, message, signature });

      const body: any = {
        tokenId,
        senderAddress: address,
        tokenType,
        signature,
      };
      if (tokenType === 'PAYG') {
        body.volumeInKB = (dataVolume || 0).toString();
      }
      this.result = await this.dataGatewayApiService.validateAccess(body);
      this.onResult.emit(this.result);
      console.log('Validation response:', this.result);
    } catch (error: any) {
      console.error('Validation failed:', error);
      this.result = {
        error: error.message || 'Unauthorized access or validation failed.',
      };
    } finally {
      this.loading = false;
    }
  }

  toggleMenu() {
    this.menuOpen = !this.menuOpen;
  }

  openAddAssetModal() {
    this.showModal = true;
  }

  resetNewCardForm() {
    this.newCard = {
      tokenType: 'SUB',
      offerId: '',
      assetName: '',
      assetDescription: '',
    };
    this.showModal = false;
  }

  saveNewCard() {
    const savedCardsRaw = localStorage.getItem('userAddedCards');
    const manuallyAddedCards = savedCardsRaw ? JSON.parse(savedCardsRaw) : [];
    manuallyAddedCards.push(this.newCard);
    localStorage.setItem('userAddedCards', JSON.stringify(manuallyAddedCards));
    this.resetNewCardForm();
    this.loadUserOffers();
  }

  clearAddedAssets() {
    localStorage.removeItem('userAddedCards');
    this.loadUserOffers();
  }
}
