import { Injectable } from '@angular/core';

// import { MetaMaskSDK } from "@metamask/sdk";
import {
  createPublicClient,
  http,
  defineChain,
  createWalletClient,
  custom,
  WalletClient,
  SignTransactionParameters,
  Address,
  GetChainIdReturnType,
} from 'viem';
import { environment } from '../environments/environment';
import { TokenType } from '../types/types';
import { Subject } from 'rxjs';

@Injectable({
  providedIn: 'root',
})
export class MetaWalletService {
  private walletClient!: WalletClient;
  public publicClient: any;
  private chain: any = defineChain(environment.walletConfig);
  private account: any;

  private accountSub = new Subject<Address>();
  public account$ = this.accountSub.asObservable();

  constructor() {
    const ethereum = (window as any).ethereum;

    if (!ethereum) {
      alert(
        'No Ethereum provider detected. Please install EVM Blockchain Wallet.',
      );
      return;
    }
    if (ethereum) {
      try {
        this.publicClient = createPublicClient({
          chain: this.chain,
          transport: http(),
        });
      } catch (error) {
        console.log(
          'Something went wrong, plase check your wallet config',
          error,
        );
      }

      //Create wallet client to interact with ledger.Sign transactions
      try {
        this.walletClient = createWalletClient({
          account: this.account,
          chain: this.chain,
          transport: custom((window as any).ethereum!),
        });
      } catch (error) {
        console.log('no metamask extension installed ', error);
      }

      this.account = this.getAccount();
      this.accountSub.next(this.account);

      (window as any).ethereum!.on('accountsChanged', (accounts: Address[]) => {
        console.log('Account changed:', accounts);
        this.account = accounts[0];
        this.accountSub.next(this.account);
      });
    }
  }

  async getChain(): Promise<GetChainIdReturnType> {
    return this.walletClient.getChainId();
  }

  // Query the browser to check if metamask is conencted if not notify user
  async checkChainId(): Promise<number> {
    const chainId = await (window as any).ethereum!.request({
      method: 'eth_chainId',
    });
    if (chainId == environment.walletConfig.id) {
      return chainId;
    } else {
      await this.walletClient.switchChain(this.chain);
      return 0;
    }
  }

  // Get wallet answer using browser window
  async getAccount(): Promise<Address> {
    const [account] = await (window as any).ethereum!.request({
      method: 'eth_requestAccounts',
    });
    return account;
  }

  async signAndSentTransaction(unsignedTx: SignTransactionParameters) {
    let rawTx: SignTransactionParameters = {
      ...unsignedTx,
      account: await this.getAccount(),
    };

    console.log(`Starting to prepare raw transaction ${JSON.stringify(rawTx)}`);

    let gas: bigint;

    try {
      gas = await this.publicClient.estimateGas({
        ...rawTx,
      });
      console.log(`Gas  for transaction is ${gas}`);
    } catch (error) {
      console.log('Error getting gas price', error);
      gas = 1000000n;
    }
    rawTx = {
      ...rawTx,
      gas: gas,
    };

    try {
      const gasPrice = await this.publicClient.getGasPrice();
      console.log(`Gas Price for transaction is ${gasPrice}`);
      rawTx = {
        ...rawTx,
        gasPrice: gasPrice,
      };
    } catch (error) {
      console.error('Error getting gas price', error);
      rawTx = {
        ...rawTx,
        gasPrice: 1000000000n as any,
      };
    }

    try {
      const nonce = await this.publicClient.getTransactionCount({
        address: rawTx.account,
      });
      console.log(`Nonce for transaction is ${nonce}`);
      rawTx = {
        ...rawTx,
        nonce: nonce,
      };
    } catch (error) {
      console.error('Error getting nonce', error);
    }

    console.log('Start sending transaction to the blockchain', rawTx);

    try {
      const tx = await this.walletClient.sendTransaction(rawTx);
      console.log('tx', tx);
      alert('Transaction was successfully sent. Tx hash: ' + tx);
    } catch (error) {
      alert('Error sending transaction. Error: ' + error);
      console.error('Error sending transaction', error);
    }
  }

  async signAccessMessage(tokenId: string, tokenType: TokenType) {
    console.log('Getting wallet account...');
    const address = await this.getAccount();
    console.log('Got address:', address);

    const nonce = new Date().toISOString().slice(0, 16);

    // Validate signature
    const message = `FAME Data Access Authentication Request

By signing this message, you confirm your identity and authorize access to the specified data resource.

Token ID: ${tokenId}
TokenType: ${tokenType}
Wallet Address: ${address}
Timestamp: ${nonce}`;

    console.log('Message to sign:', message);

    console.log('Trying to sign message...');
    const signature = await this.walletClient.signMessage({
      account: address,
      message,
    });

    console.log('Got signature:', signature);

    return { address, message, signature };
  }
}
