import { HttpClient, HttpHeaders } from '@angular/common/http';
import { environment } from '../environments/environment';
import { Injectable } from '@angular/core';
import { lastValueFrom } from 'rxjs';
import { SignTransactionParameters } from 'viem';
import { AuthTokenService } from './auth-token.service';

export interface DataAccessContractAddresses {
  contractAddressDataAccessPAYG: string;
  contractAddressDataAccessSubscription: string;
  contractAddressDataAccessPAYU: string;
}

@Injectable({
  providedIn: 'root',
})
export class TmApiService {
  constructor(
    private httpClient: HttpClient,
    private authTokenService: AuthTokenService,
  ) {}

  getApprovalTx(): Promise<SignTransactionParameters> {
    return lastValueFrom(
      this.httpClient.get<SignTransactionParameters>(
        environment.api.url +
          '/payment/approve-operator-to-spend-payment-tokens',
        this.getHttpOptions(),
      ),
    );
  }
  getSubmissionOrderTx(oid: string): Promise<SignTransactionParameters> {
    return lastValueFrom(
      this.httpClient.post<SignTransactionParameters>(
        environment.api.url + '/submissions/order',
        {
          oid,
        },
        this.getHttpOptions(),
      ),
    );
  }

  fetchContractAddresses(): Promise<DataAccessContractAddresses> {
    return lastValueFrom(
      this.httpClient.get<any>(
        environment.api.url + '/contract-info/addresses/data-access',
        this.getHttpOptions(),
      ),
    ).catch(this.handleError);
  }

  async fetchPurchasedOffers(
    userAddress: string,
  ): Promise<Record<string, string[]>> {
    const endpoint = `${environment.api.url}/purchases/offers?userAddress=${userAddress}`;

    try {
      return await lastValueFrom(
        this.httpClient.get<any>(endpoint, this.getHttpOptions()),
      );
    } catch (error) {
      return this.handleError(error);
    }
  }

  fetchTokenIdForOffer(offerId: string): Promise<string> {
    const endpoint = `${environment.api.url}/data-access-info/get-tokenid-from-offerid?offerId=${offerId}`;

    return lastValueFrom(
      this.httpClient.get(endpoint, {
        ...this.getHttpOptions(),
        responseType: 'text',
      }),
    );
  }

  handleError(error: any): Promise<any> {
    console.error('An error occurred', error);
    let message = '';

    if (error?.error?.message) {
      message = error.error.message;
    } else if (error?.message) {
      message = error.message;
    } else {
      message = JSON.stringify(error);
    }
    alert('API call error: ' + message);
    return Promise.reject(message);
  }

  private getHttpOptions() {
    return {
      headers: new HttpHeaders({
        'Content-Type': 'application/json',
        Authorization: 'Bearer ' + this.authTokenService.getAuthToken(),
        'Access-Control-Allow-Origin': '*',
      }),
    };
  }
}
