# FAME Data Access Applications - Technical Documentation


## About These Applications

This documentation describes three applications that demonstrate secure data access validation for blockchain-based data marketplaces. The applications show how data providers can validate token ownership and grant access to their datasets using cryptographic signatures and smart contract integration.

### Applications

- **data-gate**: Internal validation service for token authentication
- **data-provider-poc**: Public API gateway for token redemption
- **tm-ui-demo**: Web interface for testing and demonstration

---

## data-gate

**Purpose**: Internal microservice that validates data access tokens and processes payments  
**Framework**: NestJS + TypeScript  
**Port**: 8888

### How data-gate Works

data-gate serves as the security backbone for token validation. It verifies cryptographic signatures, confirms token ownership through blockchain queries, and processes payments for usage-based tokens.

### Key Features

- **Signature Verification**: ECDSA signature validation using ethers.js
- **Token Ownership Validation**: Real-time blockchain queries
- **Payment Processing**: Smart contract transaction execution
- **Multi-Token Support**: SUB, PAYG, and PAYU token types
- **Flexible Validation**: On-chain and off-chain validation modes

### Smart Contract Integration

data-gate integrates with blockchain smart contracts through JSON ABI files for token ownership verification, expiration checking, and payment processing.

### API Endpoints

- `GET /api/v1.0/validate/signature` - Verify message signatures
- `GET /api/v1.0/validate/access-token` - Validate token access rights
- `POST /api/withdraw-payment-tokens-PAYG` - Process PAYG payments

---

## data-provider-poc

**Purpose**: Public-facing API for data consumers to redeem access tokens  
**Framework**: NestJS + TypeScript  
**Port**: 7777

### How data-provider-poc Works

data-provider-poc acts as the public gateway between data consumers and the internal validation system. It handles token redemption requests, coordinates with data-gate for validation, and manages usage tracking.

### Key Features

- **Token Redemption**: Primary endpoint for accessing data
- **Usage Tracking**: Monitor consumption for PAYG tokens
- **Access Credential Generation**: Issue temporary data access credentials
- **Request Proxying**: Forward validation to data-gate
- **Error Handling**: Comprehensive error responses

### Integration Points

- **data-gate**: Internal validation service
- **Trading Management API**: Token metadata retrieval
- **Mock Usage Store**: In-memory usage tracking

### API Endpoints

- `POST /api/redeem-data-access` - Redeem data access tokens

---

## tm-ui-demo

**Purpose**: Web interface for testing and demonstrating the token validation system  
**Framework**: Angular 18 + TypeScript + Viem  
**Port**: 4200

### How tm-ui-demo Works

tm-ui-demo provides a complete web interface for testing token validation scenarios. It demonstrates MetaMask integration, signature generation, and data access flows.

### Key Features

- **MetaMask Integration**: Wallet connection and signature generation
- **Token Testing**: Test different token types and scenarios
- **Signature Demonstration**: Show cryptographic signing process
- **Error Simulation**: Test various failure conditions
- **Access Flow Visualization**: Complete user journey demonstration

### Test Capabilities

- **Valid Token Access**: Successful validation scenarios
- **Invalid Signatures**: Authentication failure testing
- **Expired Tokens**: Timeframe validation testing
- **Volume Limits**: PAYG consumption limit testing
- **Payment Processing**: Transaction execution testing

---

## Token Types and Validation

### Subscription (SUB) Tokens

Pre-paid tokens providing unlimited access during a valid timeframe.

**How SUB Validation Works**:

1. Verify token ownership through blockchain query
2. Check expiration timestamp
3. Grant unlimited access if valid

### Pay-As-You-Go (PAYG) Tokens

Usage-based tokens where users pay for data consumption by volume.

**How PAYG Validation Works**:

1. Verify token ownership
2. Check remaining volume capacity
3. Process payment based on requested volume
4. Track usage against token limits
5. Grant access for specified volume

### Pay-As-You-Use (PAYU) Tokens

Download based tokens providing access for dowload .

**How PAYU Validation Works**:

1. Verify token ownership
2. Check remaining downloads
3. Validate expiration timeframe
4. Track usage against limits
5. Grant access based on download count

---

## MetaMask Signature Authentication

### How Authentication Works

The applications use MetaMask signatures to prove token ownership without exposing private keys.

### Message Format

Users sign structured messages containing:

- Token identification (ID and type)
- Wallet address for verification
- Timestamp for replay protection
- Human-readable content

### How Signature Process Works

1. **Message Construction**: Create structured authentication message
2. **MetaMask Signing**: User signs message in MetaMask popup
3. **Signature Transmission**: Send signature with token data
4. **Cryptographic Verification**: Verify signature authenticity
5. **Address Recovery**: Recover signer address from signature
6. **Ownership Confirmation**: Confirm recovered address matches claimed address

### Security Features

- **ECDSA Cryptography**: Standard Ethereum signature verification
- **Private Key Protection**: Keys never leave MetaMask environment
- **Tamper Detection**: Message modification invalidates signature
- **Replay Protection**: Timestamp inclusion prevents reuse

---

## Usage Tracking (PAYG)

### How Volume Management Works

PAYG tokens require precise volume tracking to enforce consumption limits.

### How Tracking Process Works

1. **Token Metadata Retrieval**: Get volume caps from blockchain
2. **Current Usage Calculation**: Sum historical consumption
3. **Remaining Capacity**: Calculate available volume
4. **Request Validation**: Ensure requests don't exceed limits
5. **Usage Recording**: Track consumption after successful access

### Storage Implementation

- **In-Memory Store**: MockUsageStore for POC demonstration
- **Downloads Recording**: Individual usage count
- **Cumulative Tracking**: Total consumption monitoring
- **Limit Enforcement**: Real-time volume cap validation

---

## Smart Contract Integration

### How Contract Interaction Works

The applications interact with blockchain smart contracts for:

- **Token Ownership**: Verify current token holders
- **Expiration Times**: Check token validity periods
- **Payment Processing**: Execute usage-based payments
- **Token Metadata**: Retrieve token details and limits

### Key Contract Functions

- **ownerOf()**: Token ownership verification
- **getLastDataAccessExpirationTime()**: Expiration checking
- **transferFrom()**: Payment processing
- **getOfferIdInfo()**: Token metadata retrieval

### ABI Management

Smart contract interfaces are managed through JSON ABI files for each contract type.

---

## System Integration

### How Complete Request Flow Works

1. **User Interaction**: tm-ui-demo interface for token access
2. **Signature Generation**: MetaMask signature creation
3. **Request Submission**: POST to data-provider-poc
4. **Validation Coordination**: data-provider-poc calls data-gate
5. **Blockchain Verification**: Smart contract queries
6. **Payment Processing**: PAYG payment execution
7. **Usage Recording**: Consumption tracking
8. **Access Grant**: Temporary credential generation

### How Validation Strategies Work

- **On-Chain Validation**: Direct blockchain queries for real-time state
- **Off-Chain Validation**: Cached data from Trading Management API
- **Hybrid Approach**: Strategy selection based on token type

---

## Configuration

### Environment Variables

#### data-gate

- `BLOCKCHAIN_RPC_URL`: Ethereum network connection
- `VALIDATION_METHOD`: ON_CHAIN or OFF_CHAIN
- `TM_API`: Trading Management API URL
- `PROVIDER_PRIVATE_KEY`: Transaction signing key
- `JWT`: API authentication token

#### data-provider-poc

- `TM_API`: Trading Management API URL
- `JWT`: API authentication token
- `DATA_ACCESS_VALIDATOR_API`: data-gate service URL

#### tm-ui-demo

- `tmApiUrl`: Trading Management API URL
- `dataGatewayApiUrl`: data-provider-poc URL
- `blockchainRpcUrl`: Blockchain network connection

---

## Error Handling

### Common Error Scenarios

- **Invalid Signatures**: Cryptographic verification failures
- **Token Ownership Issues**: Mismatched ownership claims
- **Expiration Errors**: Tokens outside valid timeframes
- **Volume Limit Violations**: PAYG consumption exceeding capacity
- **Payment Failures**: Transaction processing errors

### Error Response Structure

Errors include HTTP status codes, descriptive messages, and relevant validation details for debugging.

---

## Testing and Demonstration

### Test Scenarios

- **Valid Token Access**: Successful validation and access grant
- **Authentication Failures**: Invalid signature scenarios
- **Expired Tokens**: Timeframe validation failures
- **Volume Limits**: PAYG consumption limit testing
- **Payment Processing**: Transaction execution and failure handling

### Integration Testing

- **End-to-End Flows**: Complete user journey simulation
- **Error Condition Handling**: Various failure scenario testing
- **Performance Validation**: Response time and throughput testing
- **Security Verification**: Signature and ownership validation testing

---

## Deployment

### Prerequisites

- Node.js v20+
- Ethereum-compatible blockchain network
- MetaMask wallet for testing
- Access to Trading Management API

### Setup Steps

1. **Install Dependencies**: npm install in each application
2. **Configure Environment**: Set required environment variables
3. **Start Services**: Launch data-gate, data-provider-poc, and tm-ui-demo
4. **Test Integration**: Verify communication between components

### Production Considerations

- **Security**: Secure private key management
- **Scalability**: Load balancing and horizontal scaling
- **Monitoring**: Logging and error tracking
- **Backup**: Usage data persistence for production systems

These applications demonstrate secure, blockchain-based data access validation suitable for integration into production data marketplace systems.
