# FAME Data Provider Integration Guide

This document provides instructions for data providers on how to integrate the FAME Data Gate application into their systems. The Data Gate is a microservice that validates data access tokens and executes on-chain operations, allowing data providers to securely offer paid access to their datasets through the FAME marketplace.

## Table of Contents

- [FAME Data Provider Integration Guide](#fame-data-provider-integration-guide)
  - [Table of Contents](#table-of-contents)
  - [Overview](#overview)
  - [Architecture](#architecture)
    - [Reference Architecture Diagram](#reference-architecture-diagram)
  - [Monorepo Implementation Reference](#monorepo-implementation-reference)
  - [Data Gate Setup Instructions](#data-gate-setup-instructions)
    - [1. Prerequisites](#1-prerequisites)
    - [2. Environment Variables](#2-environment-variables)
    - [3. Run with Docker](#3-run-with-docker)
  - [Implementation Reference Workflow](#implementation-reference-workflow)
    - [1. Fetch Customer's Purchased Offers](#1-fetch-customers-purchased-offers)
    - [2. Filter and Display Offers](#2-filter-and-display-offers)
    - [3. Get Token ID from Offer ID](#3-get-token-id-from-offer-id)
    - [4. Validate Customer Signature](#4-validate-customer-signature)
  - [Data Gate API Endpoints](#data-gate-api-endpoints)
    - [GET /api/v1.0/validate/signature](#get-apiv10validatesignature)
    - [GET /api/v1.0/validate/access-token](#get-apiv10validateaccess-token)
    - [POST /api/withdraw-payment-tokens-PAYG](#post-apiwithdraw-payment-tokens-payg)
  - [References](#references)

## Overview

The FAME Data Gate is an internal microservice that needs to be implemented by data providers. It acts as a secure gateway for data providers to manage access to their datasets. It validates data access tokens issued on the FAME marketplace and handles the necessary on-chain operations to ensure that only authorized users can access the data.

## Architecture

The integration architecture consists of two main components that data providers need to be aware of:

1.  **Data Gate**: An internal microservice that validates data access tokens and processes payments. This is the core component that data providers must run.
2.  **Data Provider App**: A public-facing service (backend, frontend, or both) that exposes an API and/or UI for customers to redeem their data access tokens and access the data. Each data provider is responsible for implementing this component according to their needs.

The Data Provider App receives requests from customers, forwards them to the Data Gate for validation, and then returns the appropriate response to the customer.

### Reference Architecture Diagram

The following diagram illustrates the reference implementation provided in the FAME TM Monorepo, showing the interaction between the different services.

```mermaid
graph TD
    A[Customer via Portal UI] --> B{Data Provider POC};
    B --> C{Data Gate};
    C --> D[FAME TM API];
    C --> E[Blockchain];
```

## Monorepo Implementation Reference

To help data providers build their consumption applications, the FAME TM Monorepo provides a full reference implementation. Developers should look at these applications to understand how to interact with the FAME ecosystem.

-   **Data Provider POC (`apps/data-provider-poc`)**: A reference backend proxy that shows how a data provider's application can communicate with the `data-gate`.
-   **TM UI Demo (`apps/tm-ui-demo`)**: A reference frontend application that demonstrates the end-to-end customer flow, from connecting a wallet to signing a data access request.

The reference implementation showcases the following features:

- Request and Read Consumer Data Access Token from Consumer Wallet
- Check Data Access Token Validity
- Check Data Access Token Ownership
- Check Data Access Token Expiry Date
- Check Data Access Token Consumption Counter
- Transfer Payment Tokens from Customer to Data Provider for PAYG Data Access Token
- Check Customer Payment Token Balance if needed
- Register Data Access Token Consumption

## Data Gate Setup Instructions

Data providers are expected to run the **Data Gate** application using the provided Docker image.

### 1. Prerequisites

- Docker

### 2. Environment Variables

Create a `.env` file and provide the following configuration for the Data Gate container:

```bash
# The RPC endpoint for the blockchain network.
BLOCKCHAIN_RPC_URL=http://162.55.94.15:8545

# The validation strategy. Can be ON_CHAIN or OFF_CHAIN.
VALIDATION_METHOD=ON_CHAIN

# The endpoint for the FAME Trading & Monetization (TM) API.
TM_API=https://tm.fame-horizon.eu/api/v1.0

# The private key of the data provider's wallet.
PROVIDER_PRIVATE_KEY=0xprovider-private-key

# A long-lived JWT for authenticating with the FAME TM API.
# Alternatively, you can implement your own mechanism to generate short-lived tokens.
JWT=your-jwt-token
```

**Validation Methods**

- `ON_CHAIN`: Uses smart contracts to verify token expiration and clearance. This is the most secure method.
- `OFF_CHAIN`: Relies on the FAME Trading Management API for validation. This method is faster but less decentralized.

### 3. Run with Docker

Data providers should pull the official pre-built image from the Harbor registry and run it as a container.

**Pull the Image**

```bash
docker pull harbor.gftinnovation.eu/fame/framework/tm/data-gate
```

**Run the Container**

```bash
docker run --env-file .env -p 8888:8888 harbor.gftinnovation.eu/fame/framework/tm/data-gate
```

The service will be available on port `8888`.

**Running All Backend Services with Docker Compose**

For a complete development setup, you can run all required backend services (FAME TM Public API, FAME TM Private API, Data Gate, and Data Provider POC) together using the provided Docker Compose file.

```bash
docker compose -f devops/docker-compose.yml up
```

**Note**: This command does not start the Portal UI (`tm-ui-demo`), which needs to be run separately.

## Implementation Reference Workflow

This section describes the end-to-end workflow as demonstrated in the `tm-ui-demo` reference application.

### 1. Fetch Customer's Purchased Offers

The UI fetches all data access offers purchased by the connected customer's wallet from the FAME Public TM API.

**Example Request:**
```
GET http://localhost:3001/api/v1.0/purchases/offers?userAddress=0xf2d72bf781c11186553884574c01175c7d205822
```

**Example Response:**
```json
{
  "SUB": ["12CwhrbhP7wwWHkpxTRqthz8e8UZN6WXmrz1gk2oXVpaARtgJLA", ...],
  "PAYG": ["1o5SPz4GQiuidbC1NmR5dY1CTGFLL3ffMvLnUvnme9L2Fmtosv", ...],
  "PAYU": ["1QvuGnoqbGVxUAiFP2yKuxVWnDSFt2k1SXcrbvzvyaojmka9Qm", ...]
}
```

### 2. Filter and Display Offers

The UI displays only the offers that:
a) Were purchased by the customer (from the API response above).
b) Are recognized by the data provider.

Currently, there is no FAME endpoint for a data provider to fetch a list of their own created offers. Therefore, the reference UI mocks this by storing a list of valid offer IDs. In a real-world scenario, the data provider would need to maintain their own database of created offers.

- **Mock Data Reference**: `apps/tm-ui-demo/src/components/data-access-cards/data-access-data.ts`

### 3. Get Token ID from Offer ID

Before validating a data access request, the Data Provider App must retrieve the unique `TokenID` (an ERC1155 identifier) that corresponds to the `OfferID`. This is done by querying the FAME TM API.

**Example Request:**
```
GET http://localhost:3001/api/v1.0/data-access-info/get-tokenid-from-offerid?offerId=12eJVWRYs6uchZMVmg9fLNrbtwAcTDwe8ELwErMwrb2Tq4iTDDj
```

**Example Response:**
```
98933654644976888979567719477680639818541285750444040794334665202904035264375
```

### 4. Validate Customer Signature

To prove ownership of the data access token, the customer must sign a message with their wallet. The Data Gate validates this signature.

The message payload must be structured precisely as follows, as the Data Gate reconstructs and verifies it. The customer has one minute to sign.

**Message Structure:**
```
FAME Data Access Authentication Request

By signing this message, you confirm your identity and authorize access to the specified data resource.

Token ID: <Token ID>
TokenType: <SUB|PAYG|PAYU>
Wallet Address: <Customer's Wallet Address>
Timestamp: <ISO 8601 Timestamp>
```

- **Signature Generation Reference**: See the `signAccessMessage` function in `apps/tm-ui-demo/src/services/metamask.service.ts`.

## Data Gate API Endpoints

The Data Gate exposes the following API endpoints for validation and payment processing:

### GET /api/v1.0/validate/signature

Verifies an ECDSA signature to ensure it matches the expected Ethereum address.

**Request Body**

```json
{
  "message": "123:0xf2d72bf781c11186553884574c01175c7d205822:SUB",
  "signature": "0x1234567890abcdef...",
  "expectedAddress": "0xf2d72bf781c11186553884574c01175c7d205822"
}
```

### GET /api/v1.0/validate/access-token

Validates a user's access to a dataset by checking token ownership, expiration, and signature.

**Request Body**

```json
{
  "tokenId": "123",
  "senderAddress": "0xf2d72bf781c11186553884574c01175c7d205822",
  "tokenType": "SUB",
  "signature": "0x1234567890abcdef..."
}
```

### POST /api/withdraw-payment-tokens-PAYG

Processes payment token withdrawals for Pay-As-You-Go (PAYG) data access.

**Query Parameters**

```json
{
  "customerAddress": "0xf2d72bf781c11186553884574c01175c7d205822",
  "amount": "1000"
}
```

## References

- **FAME TM Monorepo**: [https://gitlab.gftinnovation.eu/fame/framework/tm](https://gitlab.gftinnovation.eu/fame/framework/tm)
- **Data Gate Docker Images**: [Harbor Repository](https://harbor.gftinnovation.eu/harbor/projects/40/repositories/framework%2Ftm%2Fdata-gate/artifacts-tab)
