# FAME Trading Management - Public API Documentation

## Overview

This documentation covers all publicly accessible functions that developers can interact with when integrating with the FAME Trading Management system. Only functions that can be called by any user (without owner privileges or special access controls) are included.

---

## BaseDataAccess Contracts

### Common Public Functions (Available in DataAccessPayAsYouGo, DataAccessPayAsYouUse, DataAccessSubscription)

#### `getDataAccessTokenAmount(uint256 _assetId) → uint256`
Returns the total amount of tokens minted for a specific asset ID.

**Parameters:**
- `_assetId`: The asset identifier

**Returns:** Total amount of tokens for the asset

#### `balanceOf(address account, uint256 id) → uint256`
Returns the balance of tokens for a specific account and token ID (inherited from ERC1155).

**Parameters:**
- `account`: The address to query
- `id`: The token ID

**Returns:** Token balance

#### `isApprovedForAll(address account, address operator) → bool`
Returns whether an operator is approved for all tokens of an account (inherited from ERC1155).

**Parameters:**
- `account`: The token owner
- `operator`: The operator address

**Returns:** True if operator is approved

#### `uri(uint256 id) → string`
Returns the URI for a token ID (inherited from ERC1155).

**Parameters:**
- `id`: The token ID

**Returns:** Token URI string

---

## OfferingToken.sol

### Token Existence and Information

#### `tokenExists(string memory _oid) → bool`
Checks if an offering currently exists (not burned).

**Parameters:**
- `_oid`: The offering identifier

**Returns:** True if the offering exists

#### `historicallyTokenExists(string memory _oid) → bool`
Checks if an offering ever existed (including burned ones).

**Parameters:**
- `_oid`: The offering identifier

**Returns:** True if the offering ever existed

#### `getOfferingDataProvider(string memory _oid) → address`
Returns the data provider address for a specific offering.

**Parameters:**
- `_oid`: The offering identifier

**Returns:** Data provider address

#### `getOfferingTokenIndxDataProvider(uint256 _tokIndx) → address`
Returns the data provider address for a specific token index.

**Parameters:**
- `_tokIndx`: The token index

**Returns:** Data provider address

### Pricing and Offering Details

#### `getOfferDataAccessPrice(string memory _oid) → uint256`
Returns the price for accessing the data offering.

**Parameters:**
- `_oid`: The offering identifier

**Returns:** Price in payment tokens

#### `getOfferDurationTime(string memory _oid) → uint256`
Returns the duration time for the offering.

**Parameters:**
- `_oid`: The offering identifier

**Returns:** Duration in seconds

#### `getOfferIdInfo(string memory _oid) → OIdInfo`
Returns complete offering information.

**Parameters:**
- `_oid`: The offering identifier

**Returns:** OIdInfo struct containing:
- `dataAccessPrice`: Price for access
- `capDuration`: Duration cap
- `capDownloads`: Download limitations
- `capVolume`: Volume limitations
- `cdsTarget`: CDS target
- `cdsSl`: CDS SL
- `oid`: Original offering ID

### Asset and Offering Relationships

#### `getAssetDataProvider(string memory _assetid) → address`
Returns the data provider for a specific asset.

**Parameters:**
- `_assetid`: The asset identifier

**Returns:** Data provider address

#### `getAssetidOfOid(string memory oid) → string`
Returns the asset ID associated with an offering ID.

**Parameters:**
- `oid`: The offering identifier

**Returns:** Associated asset ID

#### `getAssetidHashOfOidHash(uint256 oidHash) → uint256`
Returns the asset ID hash for a given offering hash.

**Parameters:**
- `oidHash`: The offering hash

**Returns:** Asset ID hash

### Asset Offering Lists

#### `getNoOfAssetOids(uint256 assetidHash) → uint256`
Returns the number of offerings for an asset.

**Parameters:**
- `assetidHash`: The asset ID hash

**Returns:** Number of offerings

#### `getAssetOids(uint256 assetidHash) → uint256[]`
Returns all offering hashes for an asset.

**Parameters:**
- `assetidHash`: The asset ID hash

**Returns:** Array of offering hashes

#### `getAssetOidAtIndex(uint256 assetidHash, uint256 index) → uint256`
Returns offering hash at specific index for an asset.

**Parameters:**
- `assetidHash`: The asset ID hash
- `index`: Index in the offerings list

**Returns:** Offering hash at the specified index

### Token Information

#### `getTotalMinted() → uint256`
Returns the total number of tokens ever minted.

**Returns:** Total minted count

#### `getTotalBurned() → uint256`
Returns the total number of tokens burned.

**Returns:** Total burned count

#### `getMintedTokens() → uint256[]`
Returns array of currently existing token IDs (excludes burned tokens).

**Returns:** Array of active token IDs

#### `getHistoricallyMintedTokens() → uint256[]`
Returns array of all historically minted token IDs (includes burned tokens).

**Returns:** Array of all token IDs ever minted

#### `getHistoricallyMintedTokenAt(uint256 i) → uint256`
Returns token ID at specific index in historical list.

**Parameters:**
- `i`: Index in the historical list

**Returns:** Token ID at the specified index

### Utility Functions

#### `getIDHash(string memory _str) → uint256`
Generates hash for string identifiers.

**Parameters:**
- `_str`: String to hash

**Returns:** Hash value

#### `getOidFromHash(uint256 _oid) → string`
Returns the original offering ID from its hash.

**Parameters:**
- `_oid`: The offering hash

**Returns:** Original offering ID string

#### `getAssetidFromHash(uint256 _assetid) → string`
Returns the original asset ID from its hash.

**Parameters:**
- `_assetid`: The asset ID hash

**Returns:** Original asset ID string

### Standard ERC721 Functions

#### `ownerOf(uint256 tokenId) → address`
Returns the owner of a specific token.

**Parameters:**
- `tokenId`: The token ID

**Returns:** Owner address

#### `balanceOf(address owner) → uint256`
Returns the number of tokens owned by an address.

**Parameters:**
- `owner`: The owner address

**Returns:** Token count

#### `tokenURI(uint256 tokenId) → string`
Returns the URI for a token.

**Parameters:**
- `tokenId`: The token ID

**Returns:** Token URI

#### `tokenURIHash(string memory _oid) → string`
Returns the URI for an offering by its ID.

**Parameters:**
- `_oid`: The offering identifier

**Returns:** Token URI

---

## PaymentToken.sol

### Balance and Transfer Functions

#### `balanceOf(address account) → uint256`
Returns the token balance of an account.

**Parameters:**
- `account`: The account address

**Returns:** Token balance

#### `transfer(address to, uint256 amount) → bool`
Transfers tokens to another address (when not paused).

**Parameters:**
- `to`: Recipient address
- `amount`: Amount to transfer

**Returns:** True if successful

#### `transferFrom(address from, address to, uint256 amount) → bool`
Transfers tokens on behalf of another address (when not paused).

**Parameters:**
- `from`: Sender address
- `to`: Recipient address
- `amount`: Amount to transfer

**Returns:** True if successful

#### `approve(address spender, uint256 amount) → bool`
Approves another address to spend tokens on your behalf.

**Parameters:**
- `spender`: Address to approve
- `amount`: Amount to approve

**Returns:** True if successful

#### `allowance(address owner, address spender) → uint256`
Returns the remaining amount that spender can transfer from owner.

**Parameters:**
- `owner`: Token owner
- `spender`: Approved spender

**Returns:** Remaining allowance

### Permission Checking

#### `isPermitted(address account) → bool`
Checks if an account is permitted to transfer tokens.

**Parameters:**
- `account`: Account to check

**Returns:** True if permitted

### Token Information

#### `name() → string`
Returns the token name.

**Returns:** Token name

#### `symbol() → string`
Returns the token symbol.

**Returns:** Token symbol

#### `decimals() → uint8`
Returns the number of decimals.

**Returns:** Decimal places

#### `totalSupply() → uint256`
Returns the total token supply.

**Returns:** Total supply

---

## GovernanceFacet.sol

### Token and Balance Information

#### `getFameBourse() → address`
Returns the Fame Bourse contract address.

**Returns:** Bourse contract address

#### `getNoofCoins() → uint`
Returns the number of registered coins.

**Returns:** Number of registered coins

#### `getCoinBalance(address traderid, string memory tokensymbol) → uint`
Returns the balance of a specific token for an address.

**Parameters:**
- `traderid`: Address to check
- `tokensymbol`: Token symbol

**Returns:** Token balance

#### `getCoinNoSymbol(uint coinno) → string`
Returns the symbol for a coin number.

**Parameters:**
- `coinno`: Coin number

**Returns:** Coin symbol

#### `getCoinAddress(string memory coinsym) → (address, uint)`
Returns the contract address and token index for a coin symbol.

**Parameters:**
- `coinsym`: Coin symbol

**Returns:** 
- Contract address
- Token index

#### `getCoinNo(address coinaddr, uint tokindx) → uint`
Returns the coin number for a contract address and token index.

**Parameters:**
- `coinaddr`: Contract address
- `tokindx`: Token index

**Returns:** Coin number

#### `getCoinSymbol(address coinaddr, uint tokindx) → string`
Returns the symbol for a contract address and token index.

**Parameters:**
- `coinaddr`: Contract address
- `tokindx`: Token index

**Returns:** Coin symbol

### Data Provider Information

#### `isRegisteredDataProvider(address dataprovaddr) → bool`
Checks if a data provider is currently registered.

**Parameters:**
- `dataprovaddr`: Data provider address

**Returns:** True if registered and not expired

#### `getExpirationDataProvider(address dataprovaddr) → uint`
Returns the expiration timestamp for a data provider.

**Parameters:**
- `dataprovaddr`: Data provider address

**Returns:** Expiration timestamp

### Token Standard Detection

#### `getTokenStandard(address ta) → uint`
Determines the token standard of a contract.

**Parameters:**
- `ta`: Token contract address

**Returns:** 
- `20` for ERC20
- `721` for ERC721
- `1155` for ERC1155

#### `supportsInterface2(address accaddr, bytes4 interfaceID) → bool`
Checks if a contract supports a specific interface.

**Parameters:**
- `accaddr`: Contract address
- `interfaceID`: Interface identifier

**Returns:** True if interface is supported

---

## TradingManagementExecutorFacet.sol

### Purchase Functions

#### `purchaseAccessRightPAYG(string memory _oid, bytes memory _data, uint256 addedTime)`
Purchases Pay-As-You-Go access rights.

**Parameters:**
- `_oid`: Offering identifier
- `_data`: Additional data for the purchase
- `addedTime`: Duration to add to current time for expiration

**Requirements:**
- Offering must exist
- Caller must have sufficient payment token balance
- Caller must have approved the payment

#### `purchaseAccessRightPAYU(string memory _oid, bytes memory _data, uint256 addedTime)`
Purchases Pay-As-You-Use access rights.

**Parameters:**
- `_oid`: Offering identifier
- `_data`: Additional data for the purchase
- `addedTime`: Duration to add to current time for expiration

**Requirements:**
- Offering must exist
- Caller must have sufficient payment token balance
- Payment is transferred immediately

#### `purchaseAccessRightSubscription(string memory _oid, bytes memory _data, uint256 addedTime)`
Purchases subscription-based access rights.

**Parameters:**
- `_oid`: Offering identifier
- `_data`: Additional data for the purchase
- `addedTime`: Duration to add to current time for expiration

**Requirements:**
- Offering must exist
- Caller must have sufficient payment token balance
- Payment is transferred immediately

### Access Rights Query Functions

#### `getLastDataAccessPAYGExpirationTime(address _buyer, uint256 tokenId) → uint256`
Returns the most recent PAYG expiration time for a user and token.

**Parameters:**
- `_buyer`: Buyer address
- `tokenId`: Token identifier

**Returns:** Unix timestamp of expiration (0 if none exists)

#### `getLastDataAccessPAYUExpirationTime(address _buyer, uint256 tokenId) → uint256`
Returns the most recent PAYU expiration time for a user and token.

**Parameters:**
- `_buyer`: Buyer address
- `tokenId`: Token identifier

**Returns:** Unix timestamp of expiration (0 if none exists)

#### `getLastDataAccessSubscriptionExpirationTime(address _buyer, uint256 tokenId) → uint256`
Returns the most recent subscription expiration time for a user and token.

**Parameters:**
- `_buyer`: Buyer address
- `tokenId`: Token identifier

**Returns:** Unix timestamp of expiration (0 if none exists)

#### `getAllDataAccessExpirationsPAYG(address _buyer, uint256 tokenId) → uint256[]`
Returns all PAYG expiration timestamps for a user and token.

**Parameters:**
- `_buyer`: Buyer address
- `tokenId`: Token identifier

**Returns:** Array of expiration timestamps

#### `getAllDataAccessExpirationsPAYU(address _buyer, uint256 tokenId) → uint256[]`
Returns all PAYU expiration timestamps for a user and token.

**Parameters:**
- `_buyer`: Buyer address
- `tokenId`: Token identifier

**Returns:** Array of expiration timestamps

#### `getAllDataAccessExpirationsSubscription(address _buyer, uint256 tokenId) → uint256[]`
Returns all subscription expiration timestamps for a user and token.

**Parameters:**
- `_buyer`: Buyer address
- `tokenId`: Token identifier

**Returns:** Array of expiration timestamps

### User Offer Management

#### `getAllUserDataAccessPAYGOfferIds(address user) → uint256[]`
Returns all PAYG offer IDs for a user.

**Parameters:**
- `user`: User address

**Returns:** Array of offer IDs

#### `getAllUserDataAccessPAYUOfferIds(address user) → uint256[]`
Returns all PAYU offer IDs for a user.

**Parameters:**
- `user`: User address

**Returns:** Array of offer IDs

#### `getAllUserDataAccessSubscriptionOfferIds(address user) → uint256[]`
Returns all subscription offer IDs for a user.

**Parameters:**
- `user`: User address

**Returns:** Array of offer IDs

#### `getUserDataAccessPAYGOfferIdCount(address user) → uint256`
Returns the count of PAYG offer IDs for a user.

**Parameters:**
- `user`: User address

**Returns:** Number of offer IDs

#### `getUserDataAccessPAYUOfferIdCount(address user) → uint256`
Returns the count of PAYU offer IDs for a user.

**Parameters:**
- `user`: User address

**Returns:** Number of offer IDs

#### `getUserDataAccessSubscriptionOfferIdCount(address user) → uint256`
Returns the count of subscription offer IDs for a user.

**Parameters:**
- `user`: User address

**Returns:** Number of offer IDs

#### `getIthUserDataAccessPAYGOfferId(address user, uint256 index) → uint256`
Returns PAYG offer ID at specific index for a user.

**Parameters:**
- `user`: User address
- `index`: Index in the user's offer list

**Returns:** Offer ID at the specified index

#### `getIthUserDataAccessPAYUOfferId(address user, uint256 index) → uint256`
Returns PAYU offer ID at specific index for a user.

**Parameters:**
- `user`: User address
- `index`: Index in the user's offer list

**Returns:** Offer ID at the specified index

#### `getIthUserDataAccessSubscriptionOfferId(address user, uint256 index) → uint256`
Returns subscription offer ID at specific index for a user.

**Parameters:**
- `user`: User address
- `index`: Index in the user's offer list

**Returns:** Offer ID at the specified index

---

## TradingManagementStorageFacet.sol

### Contract Address Queries

#### `getBourseContractAddress() → address`
Returns the Bourse contract address.

**Returns:** Bourse contract address

#### `getOfferingTokenContract() → address`
Returns the OfferingToken contract address.

**Returns:** OfferingToken contract address

#### `getPaymentTokenContract() → address`
Returns the PaymentToken contract address.

**Returns:** PaymentToken contract address

#### `getPaygDataAccessContract() → address`
Returns the PAYG DataAccess contract address.

**Returns:** PAYG contract address

#### `getPayuDataAccessContract() → address`
Returns the PAYU DataAccess contract address.

**Returns:** PAYU contract address

#### `getSubscriptionDataAccessContract() → address`
Returns the Subscription DataAccess contract address.

**Returns:** Subscription contract address

---

## Escrow.sol

### Escrow Management

#### `getBourse() → address`
Returns the Bourse contract address.

**Returns:** Bourse contract address

#### `withdraw(uint256 amount, uint256 id)`
Allows users to withdraw their deposited funds for a specific asset.

**Parameters:**
- `amount`: Amount to withdraw
- `id`: Asset ID

**Requirements:**
- Amount must be greater than zero
- User must have sufficient deposited funds for the asset

#### `consumerFundsPerId(address user, uint256 assetId) → uint256`
Returns the amount of funds deposited by a user for a specific asset.

**Parameters:**
- `user`: User address
- `assetId`: Asset identifier

**Returns:** Deposited amount

---

## Integration Notes

1. **Token Approvals**: Always approve payment tokens before purchasing access rights
2. **Expiration Checking**: Use `block.timestamp` to compare with expiration times
3. **Event Monitoring**: Listen to contract events for real-time updates
4. **Error Handling**: Implement proper error handling for failed transactions
5. **Gas Estimation**: Some functions may require significant gas, especially array returns

## Common Integration Patterns

1. **Marketplace Browsing**: Use OfferingToken functions to list and display available offerings
2. **Price Checking**: Always verify prices before initiating purchases
3. **Access Validation**: Check user access rights before providing data access
4. **Balance Management**: Monitor payment token balances and permissions
5. **Expiration Tracking**: Implement systems to track and notify about expiring access rights