import { HardhatUserConfig } from 'hardhat/types';
import { existsSync } from 'node:fs';
import '@nomiclabs/hardhat-waffle';
import '@openzeppelin/hardhat-upgrades';
import '@typechain/hardhat';
import '@nomiclabs/hardhat-solhint';
import { task } from 'hardhat/config';
import path from 'path';
import fs from 'fs';

require('hardhat-contract-sizer');

import * as dotenv from 'dotenv';
import { TypechainUserConfig } from '@typechain/hardhat/dist/types';

const envFile = process.env.NODE_ENV ? `.env.${process.env.NODE_ENV}` : '.env';

if (existsSync(envFile)) {
  console.log(`Hardhat loading env file: ${envFile}`);
  dotenv.config({ path: envFile, override: true });
} else {
  console.log(`Env file not found: ${envFile}, skipping dotenv config`);
}

let ADMIN_PRIVATE_KEY = process.env.ADMIN_PRIVATE_KEY;

if (!process.env.HARDHAT_NETWORK) {
  throw new Error(
    'Compile error: Missing configuration. Ensure that HARDHAT_NETWORK is set in the environment.',
  );
}

if (!process.env.ADMIN_PRIVATE_KEY) {
  console.warn(
    'ADMIN_PRIVATE_KEY not provided. Ensure that ADMIN_PRIVATE_KEY is set in environment, Using dummy private key',
  );
  ADMIN_PRIVATE_KEY =
    '0x0000000000000000000000000000000000000000000000000000000000000000';
}

const BESU_RPC = process.env.BESU_RPC;
const INFURA_API = process.env.INFURA_API;

// =================================================================================
// TASKS
// =================================================================================

task('update-uri', 'Updates the URI for a data access token contract')
  .addParam('tokenType', 'The type of token to update (SUB, PAYG, or PAYU)')
  .addParam('newUri', 'The new URI to set for the token contract')
  .setAction(async (taskArgs, hre) => {
    const { updateUri } = await import(
      './libs/contracts/src/scripts/updateUri'
    );
    await updateUri(taskArgs.tokenType, taskArgs.newUri, hre);
  });

task('read-uri', 'Reads the current URI for a data access token contract')
  .addParam('tokenType', 'The type of token to read (SUB, PAYG, or PAYU)')
  .setAction(async (taskArgs, hre) => {
    const { readUri } = await import('./libs/contracts/src/scripts/readUri');
    await readUri(taskArgs.tokenType, hre);
  });

task('upgrade-diamond', 'Upgrades a specified facet in the specified diamond')
  .addParam('diamondName', 'The name of the diamond contract')
  .addParam('facetName', 'The name of the facet contract to upgrade')
  .setAction(async (taskArgs, hre) => {
    const {
      upgradeDiamond,
      // eslint-disable-next-line @typescript-eslint/no-var-requires
    } = require('./libs/contracts/src/scripts/upgradeDiamond');
    await upgradeDiamond(
      taskArgs.diamondName,
      taskArgs.facetName,
      {
        saveInfo: true,
      },
      hre,
    );
  });

task(
  'upgrade-transparent-proxy',
  'Upgrades a transparent proxy contract to a new implementation',
)
  .addParam(
    'proxyaddress',
    'The address of the transparent proxy contract to upgrade',
  )
  .addParam(
    'contractname',
    'The name of the new implementation contract (e.g., "MyContractV2")',
  )
  .setAction(async (taskArgs) => {
    const {
      upgradeTransparentProxyContract,
      // eslint-disable-next-line @typescript-eslint/no-var-requires
    } = require('./libs/contracts/src/scripts/upgradeTransparentProxyContract');

    await upgradeTransparentProxyContract({
      contractName: taskArgs.contractname,
      proxyContractAddress: taskArgs.proxyaddress,
    });
  });

/**
 * @task set-payment-token-permissions
 * @description Sets the permissions contract address for payment token via the GovernanceFacet.
 */
task(
  'set-payment-token-permissions',
  'Sets the permissions contract for a payment token',
)
  .addParam('diamondAddress', 'The address of the deployed Diamond')
  .addParam('permissionsAddress', 'The address of the new permissions contract')
  .addParam(
    'paymentTokenAddress',
    'The address of the payment token contract for verification',
  )
  .setAction(async (taskArgs, hre) => {
    const { diamondAddress, permissionsAddress, paymentTokenAddress } =
      taskArgs;

    console.log(
      '--- Starting Aligned Set Payment Token Permissions Script ---',
    );

    // 1. Get the signer account (must be the owner of the Diamond contract)
    const [owner] = await hre.ethers.getSigners();
    console.log(`Using account: ${owner.address} to sign the transaction.`);
    console.log(
      `This account MUST be the owner of the Diamond at ${diamondAddress}.`,
    );

    // 2. Get an instance of the GovernanceFacet attached to the Diamond's address
    const governance = await hre.ethers.getContractAt(
      'GovernanceFacet',
      diamondAddress,
      owner,
    );
    console.log(
      `Attached to GovernanceFacet interface at Diamond address: ${diamondAddress}`,
    );

    // 3. Call the new function on the GovernanceFacet
    const tx = await governance.setPaymentTokenPermissions(
      'FDE',
      permissionsAddress,
    );
    console.log(`Transaction sent with hash: ${tx.hash}`);

    console.log('Waiting for transaction confirmation...');
    await tx.wait(1);
    console.log('Transaction confirmed!');

    // 4. (Verification Step) Check the PaymentToken contract directly to confirm the change
    console.log('Verifying the change on the PaymentToken contract...');
    const paymentToken = await hre.ethers.getContractAt(
      'PaymentToken',
      paymentTokenAddress,
    );
    const actualPermissions = await paymentToken.permissionContract();

    console.log(`PaymentToken is at address: ${paymentTokenAddress}`);
    console.log(
      `✅ Permissions contract has been set to: ${actualPermissions}`,
    );

    if (actualPermissions.toLowerCase() !== permissionsAddress.toLowerCase()) {
      console.error(
        'VERIFICATION FAILED! The address was not updated correctly.',
      );
    } else {
      console.log('Verification successful!');
    }

    console.log('--- Script finished ---');
  });

task(
  'transfer-all-ownerships',
  'Transfers ownership of all contracts to a new admin',
)
  .addParam('newowner', 'The address of the new admin')
  .setAction(async ({ newowner }, hre) => {
    const [admin] = await hre.ethers.getSigners();
    const network = hre.network.name;

    const filePath = path.join(
      __dirname,
      './libs/contracts/contract-addresses.json',
    );
    const rawData = fs.readFileSync(filePath, 'utf-8');
    const contractAddresses = JSON.parse(rawData);

    if (!contractAddresses[network]) {
      throw new Error(
        `No contract addresses configured for network: ${network}`,
      );
    }

    const addresses = contractAddresses[network];

    const contracts = [
      {
        name: 'Governance Diamond',
        address: addresses.contractAddressGovernance,
      },
      {
        name: 'TradingManagement Diamond',
        address: addresses.contractAddressTradingManagement,
      },
      {
        name: 'OfferingToken',
        address: addresses.contractAddressOfferingToken,
      },
      {
        name: 'DataAccessPayAsYouGo',
        address: addresses.contractAddressDataAccessPAYG,
      },
      {
        name: 'DataAccessPayAsYouUse',
        address: addresses.contractAddressDataAccessPAYU,
      },
      {
        name: 'DataAccessSubscription',
        address: addresses.contractAddressDataAccessSubscription,
      },
      { name: 'Escrow', address: addresses.contractAddressEscrow },
    ];

    for (const contract of contracts) {
      console.log(
        `Transferring ownership of ${contract.name} at ${contract.address}`,
      );
      const instance = await hre.ethers.getContractAt(
        'Ownable',
        contract.address,
      );

      const currentOwner = await instance.owner();
      if (currentOwner.toLowerCase() !== admin.address.toLowerCase()) {
        console.warn(
          `Skipping ${contract.name}. Current owner is ${currentOwner}`,
        );
        continue;
      }

      const tx = await instance.connect(admin).transferOwnership(newowner);
      await tx.wait(1);

      const updatedOwner = await instance.owner();
      console.log(`${contract.name} ownership transferred to ${updatedOwner}`);
    }

    console.log('All ownership transfers completed.');
  });

// =================================================================================
// CONFIG
// =================================================================================

const validNetworks = ['besu', 'sepolia', 'localhost', 'hardhat'];
if (!process.env.HARDHAT_NETWORK) {
  console.error('HARDHAT_NETWORK not set');
} else if (validNetworks.includes(process.env.HARDHAT_NETWORK)) {
  console.log(`HARDHAT_NETWORK env set up as: ${process.env.HARDHAT_NETWORK}`);
} else {
  console.error(
    `HARDHAT_NETWORK not set to a valid network. Use one of: ${validNetworks.join(
      ', ',
    )}`,
  );
  process.exit(1);
}

const config: HardhatUserConfig & { typechain: TypechainUserConfig } & {
  contractSizer: Record<string, any>;
} = {
  defaultNetwork: process.env.HARDHAT_NETWORK,
  networks: {
    besu: {
      url: `http://${BESU_RPC}/`,
      accounts: [ADMIN_PRIVATE_KEY],
    },
    sepolia: {
      url: `https://sepolia.infura.io/v3/${INFURA_API}`,
      accounts: [ADMIN_PRIVATE_KEY],
    },
    //local environment
    localhost: {
      url: 'http://127.0.0.1:8545',
      accounts: [
        ADMIN_PRIVATE_KEY,
        '0xac0974bec39a17e36ba4a6b4d238ff944bacb478cbed5efcae784d7bf4f2ff80',
        '0x59c6995e998f97a5a0044966f0945389dc9e86dae88c7a8412f4603b6b78690d',
        '0x5de4111afa1a4b94908f83103eb1f1706367c2e68ca870fc3fb9a804cdab365a',
      ],
    },
    hardhat: {
      // loggingEnabled: !isTest,
      accounts: [
        {
          privateKey: ADMIN_PRIVATE_KEY,
          balance: '100000000000000000000000',
        },
        {
          privateKey:
            '0xac0974bec39a17e36ba4a6b4d238ff944bacb478cbed5efcae784d7bf4f2ff80',
          balance: '100000000000000000000000',
        },
        {
          privateKey:
            '0x59c6995e998f97a5a0044966f0945389dc9e86dae88c7a8412f4603b6b78690d',
          balance: '100000000000000000000000',
        },
        {
          privateKey:
            '0x5de4111afa1a4b94908f83103eb1f1706367c2e68ca870fc3fb9a804cdab365a',
          balance: '100000000000000000000000',
        },
      ],
    },
  },
  solidity: {
    version: '0.8.17',
    settings: {
      optimizer: {
        enabled: true,
        runs: 200,
      },
    },
  },
  mocha: {
    timeout: 600000,
  },
  paths: {
    sources: './libs/contracts/src/contracts/',
    tests: './libs/contracts/src/contracts/',
    cache: './cache',
    artifacts: './libs/contracts/artifacts',
  },
  typechain: {
    outDir: './libs/contracts/types',
    target: 'ethers-v5',
  },
  contractSizer: {
    alphaSort: true,
    disambiguatePaths: false,
    runOnCompile: false,
    strict: true,
    only: [],
  },
};

export default config;
