import {
  Catch,
  ExceptionFilter,
  ArgumentsHost,
  Logger,
  BadRequestException,
} from '@nestjs/common';
import { HttpException, HttpStatus } from '@nestjs/common';
import { ethers } from 'ethers';

interface ErrorObject {
  reason: string;
  code: string;
  error: {
    reason: string;
    code: string;
    method: string;
    transaction: {
      from: string;
      to: string;
      data: string;
      [key: string]: any;
    };
    error: {
      code: number;
      data: string;
      message: string;
      [key: string]: any;
    };
    [key: string]: any;
  };
  [key: string]: any;
}

@Catch()
export class GlobalExceptionFilter implements ExceptionFilter {
  private logger = new Logger('GlobalExceptionFilter');
  catch(exception: any, host: ArgumentsHost) {
    const ctx = host.switchToHttp();
    const response = ctx.getResponse();
    // const request = ctx.getRequest();

    const status =
      exception instanceof HttpException ||
      exception instanceof BadRequestException
        ? exception.getStatus()
        : HttpStatus.INTERNAL_SERVER_ERROR;

    let message = '';

    if (exception['response'] && exception['response']['message']) {
      message = exception['response']['message'];
    } else if (exception instanceof HttpException) {
      message = exception.getResponse().toString();
    } else {
      if (exception?.error?.error?.error?.error) {
        message = exception.error.error.error.error.toString();
      } else if (exception?.error?.error?.error) {
        message = exception.error.error.error.toString();
      } else if (exception?.error?.error) {
        message = exception.error.error.toString();
      } else if (exception?.error) {
        message = exception.error.toString();
      } else {
        message = exception.toString();
      }
    }

    if (message.includes('reverted')) {
      message =
        message + '. Decoded revert reason: ' + this.decodeETHError(exception);
    }

    this.logger.error(message);
    this.logger.error(exception);

    response.status(status).json({
      statusCode: status,
      timestamp: new Date().toISOString(),
      message,
    });
  }

  private decodeETHError(errorObject: ErrorObject): string {
    try {
      // Extract the revert data
      let revertData: string;

      if (errorObject.error?.error?.error?.error?.data) {
        revertData = errorObject.error.error.error.error.data;
      } else if (errorObject.error?.error?.error?.data) {
        revertData = errorObject.error.error.error.data;
      } else if (errorObject.error?.error?.data) {
        revertData = errorObject.error.error.data;
      } else if (errorObject.error?.data) {
        revertData = errorObject.error.data;
      }

      if (!revertData) {
        return 'No revert data found';
      }
      if (typeof revertData !== 'string') {
        if ((revertData as any).message) {
          return (revertData as any).message;
        } else {
          return (revertData as any).toString();
        }
      }

      // The first 4 bytes are the function selector, which we skip
      const errorBytes = revertData.toString().startsWith('0x')
        ? revertData.slice(10)
        : revertData.slice(8);

      // Decode the revert reason using ethers.js
      const reason = ethers.utils.defaultAbiCoder.decode(
        ['string'],
        '0x' + errorBytes,
      );

      return reason[0];
    } catch (error) {
      console.error('Failed to decode ETH error:', error);
      return 'Decoding failed';
    }
  }
}
