import { Logger } from '@nestjs/common';
import { IsDefined, IsIn, IsString, validateSync } from 'class-validator';
import * as dotenv from 'dotenv';

abstract class Config {
  private readonly logger = new Logger(Config.name);

  constructor() {
    this.loadEnvFile();
    this.setVariables();
    this.validateEnv();
  }

  protected abstract setVariables(): void;

  private loadEnvFile(): void {
    const envFile = process.env.NODE_ENV
      ? `.env.${process.env.NODE_ENV}`
      : '.env';

    const result = dotenv.config({ path: envFile });

    if (result.error) {
      console.warn(`Env file not found: ${envFile}, skipping dotenv config`);
    } else {
      console.log(`Successfully loaded  env file: ${envFile}`);
    }
  }

  private validateEnv(): void {
    const errors = validateSync(this);
    if (errors.length > 0) {
      this.logger.error({
        ctx: 'validateConfig',
        error: errors.map((e) => JSON.stringify(e.constraints)).join(', '),
      });
      console.error(
        'Validation failed. Ensure all required environment variables are set.',
      );
      process.exit(1);
    }
  }
}

export class AppConfig extends Config {
  @IsString()
  @IsDefined()
  offeringEndpoint: string;

  @IsString()
  @IsDefined()
  besuRpc: string;

  @IsString()
  @IsDefined()
  adminPrivateKey: string;

  @IsString()
  @IsDefined()
  infuraAPi: string;

  @IsString()
  @IsDefined()
  hardhatNetwork: string;

  @IsString()
  @IsDefined()
  jwtSecret: string;

  @IsString()
  @IsDefined()
  @IsIn(['LOCAL', 'ENDPOINT'])
  authMethod: string;

  @IsString()
  @IsDefined()
  authUrl: string;

  protected setVariables(): void {
    this.besuRpc = process.env.BESU_RPC;
    this.adminPrivateKey = process.env.ADMIN_PRIVATE_KEY;
    this.infuraAPi = process.env.INFURA_API;
    this.offeringEndpoint = process.env.PT_CONFIRM_OFFERING_ENDPOINT;
    this.hardhatNetwork = process.env.HARDHAT_NETWORK;
    this.jwtSecret = process.env.JWT_SECRET;
    this.authMethod = process.env.AUTH_METHOD;
    this.authUrl = process.env.AUTH_URL;
  }
}
