import {
  IsString,
  IsNotEmpty,
  IsNumberString,
  IsBooleanString,
  MaxLength,
  MinLength,
  IsEthereumAddress,
  IsOptional,
  IsEnum,
} from 'class-validator';
import { ApiProperty } from '@nestjs/swagger';

export enum TokenType {
  SUB = 'SUB',
  PAYU = 'PAYU',
  PAYG = 'PAYG',
}

export class GetAssetBalance {
  @ApiProperty()
  @IsNotEmpty()
  @IsEthereumAddress()
  userAddress: string;

  @ApiProperty()
  @IsNotEmpty()
  @IsNumberString()
  @MaxLength(20)
  assetId: number;
}

export class ApproveOperatorToSpendDataAssetDto {
  @ApiProperty()
  @IsNotEmpty()
  @IsEthereumAddress()
  userAddress: string;

  @ApiProperty()
  @IsNotEmpty()
  @MaxLength(64)
  @MinLength(64)
  privateKey: string;

  @ApiProperty()
  @IsBooleanString()
  approved: boolean;
}

export class GetIsDataAssetApprovedForOperatorDto {
  @ApiProperty()
  @IsNotEmpty()
  @IsEthereumAddress()
  userAddress: string;
}

export class TransferDataAssetDto {
  @ApiProperty()
  @IsNotEmpty()
  @IsEthereumAddress()
  userAddress: string;

  @ApiProperty()
  @IsNotEmpty()
  @MaxLength(64)
  @MinLength(64)
  privateKey: string;

  @ApiProperty()
  @IsString()
  @IsNotEmpty()
  @MaxLength(20)
  transferTo: string;

  @ApiProperty()
  @IsNotEmpty()
  @IsNumberString()
  @MaxLength(20)
  assetId: number;
}

export class PurchaseAccessRightDto {
  @ApiProperty({
    description: 'unique ID of offering',
  })
  @IsNotEmpty()
  oid: string;
}

export class PurchaseAccessRightAndMineDto extends PurchaseAccessRightDto {
  @ApiProperty({
    description:
      'Private key of the buyer account used for signing transaction',
  })
  @IsNotEmpty()
  @MaxLength(64)
  @MinLength(64)
  privateKey: string;
}

export class SetDataProvidersDto {
  @ApiProperty()
  @IsNotEmpty()
  @IsEthereumAddress()
  dataProviderAddress: string;

  @ApiProperty()
  @IsBooleanString()
  hasRight: boolean;
}

export class ListClearedItemsDto {
  @ApiProperty()
  @IsNotEmpty()
  @IsEthereumAddress()
  tradingAccount: string;
}

export class CheckClearanceReq {
  @ApiProperty({
    type: 'string',
    description: 'generated asset id during asset creation',
  })
  @IsNotEmpty()
  @IsNumberString()
  assetId: string;

  @ApiProperty({
    type: 'string',
    isArray: true,
    description: 'List of account public addresses in an array',
  })
  @IsNotEmpty()
  @IsEthereumAddress({ each: true })
  addresses: string[];

  @ApiProperty({
    type: 'string',
    description: 'Token type: PAYU, PAYG, SUB',
  })
  @IsNotEmpty()
  tokenType: string;
}

export class UserAddressOfferingIdDto {
  @ApiProperty({
    type: 'string',
    description: 'User ethereum address',
  })
  @IsNotEmpty()
  @IsEthereumAddress()
  userAddress: string;

  @ApiProperty({
    type: 'string',
    description: 'Purchased data access offering id',
  })
  @IsNotEmpty()
  @IsString()
  tokenId: string;
}

export class BurnAssetDto {
  @ApiProperty()
  @IsNotEmpty()
  @IsEthereumAddress()
  userAddress: string;

  @ApiProperty()
  @IsNotEmpty()
  @MaxLength(64)
  @MinLength(64)
  privateKey: string;

  @ApiProperty()
  @IsNotEmpty()
  @IsEthereumAddress()
  assetToBurnOwnerAddress: string;

  @ApiProperty()
  @IsString()
  @IsNotEmpty()
  @MaxLength(20)
  assetToBurnOwner: string;

  @ApiProperty()
  @IsNotEmpty()
  @IsNumberString()
  @MaxLength(20)
  assetId: number;
}

export class AccessAssetDto {
  @ApiProperty()
  @IsNotEmpty()
  @IsEthereumAddress()
  userAddress: string;

  @ApiProperty()
  @IsNotEmpty()
  @IsNumberString()
  @MaxLength(20)
  assetId: number;
}
