import { HttpException, HttpStatus, Injectable, Logger } from '@nestjs/common';
import { CommonUtils, Contracts } from '../../utils/common.utils';
import { ethers, Wallet, BigNumber } from 'ethers';
import * as dotenv from 'dotenv';
import { PurchaseAccessRightAndMineDto } from './data-access-dto/data-access.dto';
import { OfferingService } from '../offering/offering.service';

dotenv.config();

const ADMIN_PRIVATE_KEY = process.env.ADMIN_PRIVATE_KEY;

@Injectable()
export class DataAccessService {
  private contracts: Contracts;

  logger = new Logger(DataAccessService.name);

  constructor(
    private commonUtils: CommonUtils,
    private offeringService: OfferingService,
  ) {
    this.contracts = this.commonUtils.getContracts();
  }

  async getHasAccessRight(
    userAddress: string,
    assetId: number,
  ): Promise<boolean> {
    const amount: ethers.BigNumber =
      await this.contracts.contractDataAccessSubscription.balanceOf(
        userAddress,
        assetId,
      );

    if (amount.gt(0)) {
      return true;
    }

    return false;
  }

  async approveOperatorToSpendDataAsset(
    privateKey: string,
    approved: boolean,
  ): Promise<string> {
    const signer = new ethers.Wallet(privateKey, this.commonUtils.provider);
    // creating, signing and sending the transaction
    const tx = await this.contracts.contractDataAccessSubscription
      .connect(signer)
      .setApprovalForAll(this.contracts.bourseContract.address, approved);

    const receipt = await tx.wait();
    this.logger.log(
      `Approved ${signer.address} contractDataAccessSubscription by setApprovalForAll method to spend tokens`,
    );

    return receipt.transactionHash;
  }

  async getLastDataAccessSubscriptionExpirationTime(
    userAddress: string,
    tokenId: string,
  ): Promise<number> {
    const expirationTime: BigNumber =
      await this.contracts.contractTradingManagementExecutor.getLastDataAccessSubscriptionExpirationTime(
        userAddress,
        tokenId,
      );

    return expirationTime.toNumber();
  }

  async getLastDataAccessPAYGExpirationTime(
    userAddress: string,
    tokenId: string,
  ): Promise<number> {
    const expirationTime: BigNumber =
      await this.contracts.contractTradingManagementExecutor.getLastDataAccessPAYGExpirationTime(
        userAddress,
        tokenId,
      );

    return expirationTime.toNumber();
  }

  async getLastDataAccessPAYUExpirationTime(
    userAddress: string,
    tokenId: string,
  ): Promise<number> {
    const expirationTime: BigNumber =
      await this.contracts.contractTradingManagementExecutor.getLastDataAccessPAYUExpirationTime(
        userAddress,
        tokenId,
      );

    return expirationTime.toNumber();
  }

  async getAllDataAccessExpirationsPAYG(
    userAddress: string,
    tokenId: string,
  ): Promise<number[]> {
    const rawExpirations =
      await this.contracts.contractTradingManagementExecutor.getAllDataAccessExpirationsPAYG(
        userAddress,
        tokenId,
      );
    return rawExpirations.map((bn) => bn.toNumber());
  }

  async getAllDataAccessExpirationsPAYU(
    userAddress: string,
    tokenId: string,
  ): Promise<number[]> {
    const rawExpirations =
      await this.contracts.contractTradingManagementExecutor.getAllDataAccessExpirationsPAYU(
        userAddress,
        tokenId,
      );
    return rawExpirations.map((bn) => bn.toNumber());
  }

  async getAllDataAccessExpirationsSubscription(
    userAddress: string,
    tokenId: string,
  ): Promise<number[]> {
    const rawExpirations =
      await this.contracts.contractTradingManagementExecutor.getAllDataAccessExpirationsSubscription(
        userAddress,
        tokenId,
      );

    return rawExpirations.map((bn) => bn.toNumber());
  }

  async getUserDataAccessOfferIds(
    userAddress: string,
    tokenType?: 'SUB' | 'PAYG' | 'PAYU',
  ): Promise<Record<string, string[]>> {
    const processHashesToOids = (hashes) =>
      Promise.all(hashes.map((bn) => this.offeringService.getOidFromHash(bn)));

    if (tokenType) {
      let hashes;

      if (tokenType === 'SUB') {
        hashes =
          await this.contracts.contractTradingManagementExecutor.getAllUserDataAccessSubscriptionOfferIds(
            userAddress,
          );
      } else if (tokenType === 'PAYG') {
        hashes =
          await this.contracts.contractTradingManagementExecutor.getAllUserDataAccessPAYGOfferIds(
            userAddress,
          );
      } else {
        // tokenType is 'PAYU'
        hashes =
          await this.contracts.contractTradingManagementExecutor.getAllUserDataAccessPAYUOfferIds(
            userAddress,
          );
      }

      const offerIds = await processHashesToOids(hashes);
      return { [tokenType]: offerIds };
    }

    const [subHashes, paygHashes, payuHashes] = await Promise.all([
      this.contracts.contractTradingManagementExecutor.getAllUserDataAccessSubscriptionOfferIds(
        userAddress,
      ),
      this.contracts.contractTradingManagementExecutor.getAllUserDataAccessPAYGOfferIds(
        userAddress,
      ),
      this.contracts.contractTradingManagementExecutor.getAllUserDataAccessPAYUOfferIds(
        userAddress,
      ),
    ]);

    const [subOids, paygOids, payuOids] = await Promise.all([
      processHashesToOids(subHashes),
      processHashesToOids(paygHashes),
      processHashesToOids(payuHashes),
    ]);

    return {
      SUB: subOids,
      PAYG: paygOids,
      PAYU: payuOids,
    };
  }
  async getUserDataAccessTokenIds(
    userAddress: string,
    tokenType?: 'SUB' | 'PAYG' | 'PAYU',
  ): Promise<Record<string, string[]>> {
    if (tokenType) {
      let tokenIdsAsBigNumbers;

      if (tokenType === 'SUB') {
        tokenIdsAsBigNumbers =
          await this.contracts.contractTradingManagementExecutor.getAllUserDataAccessSubscriptionOfferIds(
            userAddress,
          );
      } else if (tokenType === 'PAYG') {
        tokenIdsAsBigNumbers =
          await this.contracts.contractTradingManagementExecutor.getAllUserDataAccessPAYGOfferIds(
            userAddress,
          );
      } else {
        // tokenType is 'PAYU'
        tokenIdsAsBigNumbers =
          await this.contracts.contractTradingManagementExecutor.getAllUserDataAccessPAYUOfferIds(
            userAddress,
          );
      }

      const tokenIds = tokenIdsAsBigNumbers.map((bn) => bn.toString());
      return { [tokenType]: tokenIds };
    }

    const [subTokenHashes, paygTokenHashes, payuTokenHashes] =
      await Promise.all([
        this.contracts.contractTradingManagementExecutor.getAllUserDataAccessSubscriptionOfferIds(
          userAddress,
        ),
        this.contracts.contractTradingManagementExecutor.getAllUserDataAccessPAYGOfferIds(
          userAddress,
        ),
        this.contracts.contractTradingManagementExecutor.getAllUserDataAccessPAYUOfferIds(
          userAddress,
        ),
      ]);

    return {
      SUB: subTokenHashes.map((bn) => bn.toString()),
      PAYG: paygTokenHashes.map((bn) => bn.toString()),
      PAYU: payuTokenHashes.map((bn) => bn.toString()),
    };
  }

  async getIsDataAssetApprovedForOperator(
    userAddress: string,
  ): Promise<boolean> {
    const isApproved: boolean =
      await this.contracts.contractDataAccessSubscription.isApprovedForAll(
        userAddress,
        this.contracts.bourseContract.address,
      );

    return isApproved;
  }

  // async setDataProviders(
  //   dataProviderAddress: string,
  //   hasRight: boolean,
  // ): Promise<string> {
  //   // creating custom data to log to transaction
  //   const data = {
  //     updatedDataProvider: dataProviderAddress,
  //     dataProviderHasRights: hasRight,
  //   };

  //   const jsonData: string = JSON.stringify(data);
  //   const dataAsBytes: string = ethers.utils.hexlify(
  //     ethers.utils.toUtf8Bytes(jsonData),
  //   );

  //   const ownerSigner = new ethers.Wallet(
  //     ADMIN_PRIVATE_KEY,
  //     this.commonUtils.provider,
  //   );

  //   this.contractDataAccess = this.contractDataAccess.connect(ownerSigner);

  //   const tx = await this.contractDataAccess.setDataProviders(
  //     dataProviderAddress,
  //     hasRight,
  //     dataAsBytes,
  //   );

  //   const receipt = await tx.wait();

  //   return receipt.transactionHash;
  // }

  async listClearedItems(tradingAccount: string): Promise<string[]> {
    //checks whether the OID is valid by only taking the minted (not burned) indexes of offerings
    const mintedTokens =
      await this.contracts.contractOfferingToken.getMintedTokens();

    const ownedTokens = [];

    for (let i = 0; i < mintedTokens.length; i++) {
      const tokenIdString = mintedTokens[i].toString();
      const tokenId = BigInt(tokenIdString);

      //checking in each data access contract whether the user has access rights (optimization: only one of these can be true at a time so could have a commonutils function returning the correct dataaccesscontract based on the business model (based on the saved idinfo in the offering contract))
      let balance = await this.contracts.contractDataAccessPAYG.balanceOf(
        tradingAccount,
        tokenId,
      );

      balance = (
        await this.contracts.contractDataAccessPAYU.balanceOf(
          tradingAccount,
          tokenId,
        )
      ).add(balance);

      balance = (
        await this.contracts.contractDataAccessSubscription.balanceOf(
          tradingAccount,
          tokenId,
        )
      ).add(balance);

      if (balance.gt(0)) {
        ownedTokens.push(tokenId);
      }
    }

    if (ownedTokens.length == 0) return [];

    const ownedTokensString = ownedTokens.map((bigIntValue) =>
      bigIntValue.toString(),
    );

    return ownedTokensString;
  }

  async checkClearance(
    assetId: string,
    addresses: string[],
    tokenType: string,
  ): Promise<boolean> {
    const arrayLength = addresses.length;
    const addressArray = Array.isArray(addresses)
      ? addresses
      : Array(arrayLength).fill(addresses);
    const assetArray = Array(addresses.length).fill(assetId);

    let balances;

    switch (tokenType) {
      case 'SUB':
        balances =
          await this.contracts.contractDataAccessSubscription.balanceOfBatch(
            addressArray,
            assetArray,
          );
        break;
      case 'PAYG':
        balances = await this.contracts.contractDataAccessPAYG.balanceOfBatch(
          addressArray,
          assetArray,
        );
        break;
      case 'PAYU':
        balances = await this.contracts.contractDataAccessPAYU.balanceOfBatch(
          addressArray,
          assetArray,
        );
        break;
      default:
        throw new Error(`Unsupported tokenType: ${tokenType}`);
    }

    for (let i = 0; i < balances.length; i++) {
      if (balances[i].gt(0)) {
        return true;
      }
    }
    return false;
  }

  async purchaseAccessRight(oid: string): Promise<string> {
    this.logger.log(`Start process of purchasing offer ${oid}`);
    return this.purchaseAccessRightCommon({
      oid,
      singerPrivateKey: null,
    });
  }

  async purchaseAccessRightAndMine(
    params: PurchaseAccessRightAndMineDto,
  ): Promise<string> {
    this.logger.log(
      `Start process of purchasing and mining offer ${params.oid}`,
    );
    return this.purchaseAccessRightCommon({
      oid: params.oid,
      singerPrivateKey: params.privateKey,
    });
  }

  private async purchaseAccessRightCommon({
    oid,
    singerPrivateKey,
  }: {
    oid: string;
    singerPrivateKey: string | null;
  }): Promise<string> {
    const exists = await this.contracts.contractOfferingToken.tokenExists(oid);
    if (!exists) {
      throw new HttpException(
        `Asset with oid ${oid} does not exist`,
        HttpStatus.NOT_FOUND,
      );
    }

    const data = { purchasedAssetId: oid };
    let idInfo;

    try {
      idInfo = await this.contracts.contractOfferingToken.getOfferIdInfo(oid);
    } catch (error) {
      this.logger.error(`Geetting offering Id Info Error for ID ${oid}`);
      throw new HttpException(error, 500);
    }

    const jsonData: string = JSON.stringify(data);
    const dataAsBytes: string = ethers.utils.hexlify(
      ethers.utils.toUtf8Bytes(jsonData),
    );
    let signer: Wallet;

    if (singerPrivateKey) {
      signer = new ethers.Wallet(singerPrivateKey, this.commonUtils.provider);
    } else {
      signer = new ethers.Wallet(ADMIN_PRIVATE_KEY, this.commonUtils.provider);
    }

    const handleTransactionError = async (error: any) => {
      this.logger.error(
        `Handling error for asset ${oid} while calling purchase access right function:`,
      );
      this.logger.error(error);

      const allowance = await this.contracts.contractPaymentToken.allowance(
        signer.address,
        this.contracts.bourseContract.address,
      );
      if (allowance.lt(idInfo.dataAccessPrice)) {
        throw new HttpException(
          `Insufficient allowance for spending FDE tokens on behalf of operator when purchasing asset ${oid}. Current allowance: ${this.commonUtils.weiToDecimals(
            allowance,
          )}, min. required: ${this.commonUtils.weiToDecimals(
            idInfo.dataAccessPrice,
          )}`,
          400,
        );
      }

      throw error;
    };

    let gasLimit = '3000000';
    let txResponse;
    const contractTradingManagementConnected = singerPrivateKey
      ? this.contracts.contractTradingManagementExecutor.connect(signer)
      : this.contracts.contractTradingManagementExecutor.connect(signer)
          .populateTransaction;

    const addedTimeInSeconds =
      idInfo.capDuration > 0
        ? Math.floor(Number(idInfo.capDuration) / 1000)
        : 365 * 86400; // Default to 365 days if capDuration is 0

    if (idInfo.capDownloads === '' && idInfo.capVolume === '') {
      // Subscription
      this.logger.log(
        `Purchasing offer ${oid} as SUBSCRIPTION (capDuration is not 0) with data: ${dataAsBytes}`,
      );

      gasLimit = (
        await this.contracts.contractTradingManagementExecutor.estimateGas
          .purchaseAccessRightSubscription(oid, dataAsBytes, addedTimeInSeconds)
          .catch((error) => {
            this.logger.warn(
              'Error while estimating gas limit for purchaseAccessRightSubscription. Setting up default value',
            );
            return 3000000;
          })
      ).toString();

      this.logger.log(
        `Gas limit for purchaseAccessRightSubscription: ${gasLimit}`,
      );

      try {
        txResponse =
          await contractTradingManagementConnected.purchaseAccessRightSubscription(
            oid,
            dataAsBytes,
            addedTimeInSeconds,
            {
              gasLimit: BigNumber.from(gasLimit),
            },
          );
      } catch (error) {
        await handleTransactionError(error);
      }
    } else if (idInfo.capDownloads != '') {
      // PAYU
      this.logger.log(
        `Purchasing offer ${oid} as PAYU (capDownloads is not empty)`,
      );

      try {
        gasLimit = (
          await this.contracts.contractTradingManagementExecutor.estimateGas
            .purchaseAccessRightPAYU(oid, dataAsBytes, addedTimeInSeconds)
            .catch((error) => {
              this.logger.warn(
                'Error while estimating gas limit for purchaseAccessRightPAYU. Setting up default value',
              );
              return 3000000;
            })
        ).toString();

        txResponse =
          await contractTradingManagementConnected.purchaseAccessRightPAYU(
            oid,
            dataAsBytes,
            addedTimeInSeconds,
          );
      } catch (error) {
        await handleTransactionError(error);
      }
    } else if (idInfo.capVolume) {
      // PAYG
      this.logger.log(
        `Purchasing offer ${oid} as PAYG (capVolume is not empty)`,
      );
      try {
        gasLimit = (
          await this.contracts.contractTradingManagementExecutor.estimateGas
            .purchaseAccessRightPAYG(oid, dataAsBytes, addedTimeInSeconds)
            .catch((error) => {
              this.logger.warn(
                'Error while estimating gas limit for purchaseAccessRightPAYG. Setting up default value',
              );
              return 3000000;
            })
        ).toString();

        txResponse =
          await contractTradingManagementConnected.purchaseAccessRightPAYG(
            oid,
            dataAsBytes,
            addedTimeInSeconds,
          );
      } catch (error) {
        await handleTransactionError(error);
      }
    } else {
      this.logger.error(
        `Offer ${oid} does not have proper info parameters. Info: ${idInfo}`,
      );
      throw new HttpException(
        `Could not establish proper business model for offer ${oid} because id does not have any cap defined`,
        400,
      );
    }
    if (singerPrivateKey) {
      this.logger.log('Transaction hash:', txResponse?.hash);
      // Wait for the transaction to be mined
      try {
        const receipt = await txResponse.wait();
        this.logger.log('Transaction was mined in block:', receipt.blockNumber);
        return receipt.transactionHash;
      } catch (e) {
        this.logger.error(
          `Failed to mine the transaction to purchaseAccessRightSubscription for asset ${oid}`,
          e,
        );
        throw e;
      }
    } else {
      delete txResponse.from;
      txResponse.gasLimit = gasLimit;
      const unsignedTxString = JSON.stringify(txResponse, null, 2);

      this.logger.log(
        `Unsigned transaction for purchase access right for offer ${oid} created`,
        unsignedTxString,
      );
      return `${unsignedTxString}`;
    }
  }

  // async burnAccessToken(
  //   userAddress: string,
  //   privateKey: string,
  //   burnTokenOwnerAddress: string,
  //   tokenId: number,
  // ): Promise<string> {
  //   // creating custom data to log to transaction
  //   const data = {
  //     functionCaller: userAddress,
  //     assetOwner: burnTokenOwnerAddress,
  //     destroyedAssetId: tokenId,
  //   };

  //   const jsonData: string = JSON.stringify(data);
  //   const dataAsBytes: string = ethers.utils.hexlify(
  //     ethers.utils.toUtf8Bytes(jsonData),
  //   );

  //   const signer = new ethers.Wallet(privateKey, this.commonUtils.provider);
  //   this.contractDataAccess = this.contractDataAccess.connect(signer);

  //   const tx = await this.contractDataAccess.burnAccessToken(
  //     burnTokenOwnerAddress,
  //     tokenId,
  //     dataAsBytes,
  //   );

  //   const receipt = await tx.wait();
  //   // TODO: add balane of endpoint
  //   return receipt.transactionHash;
  // }
}
