import {
  Injectable,
  BadRequestException,
  HttpException,
  HttpStatus,
  Logger,
} from '@nestjs/common';
import { CommonUtils } from '../../utils/common.utils';
import { ethers, BigNumber } from 'ethers';
import * as dotenv from 'dotenv';

dotenv.config();

const ADMIN_PRIVATE_KEY = process.env.ADMIN_PRIVATE_KEY;

@Injectable()
export class GovernanceService {
  private contractGovernance: ethers.Contract;
  private logger = new Logger(GovernanceService.name);

  constructor(private commonUtils: CommonUtils) {
    const contracts = this.commonUtils.getContracts();
    this.contractGovernance = contracts.contractGovernance;
  }

  async getTradingAccount(tid: string, token: string): Promise<any> {
    if (!this.commonUtils.isValidEthAddress(tid)) {
      throw new HttpException(
        'Invalid ethereum address',
        HttpStatus.BAD_REQUEST,
      );
    }

    const checkedsumdaddr = ethers.utils.getAddress(tid);

    let tidBalance = ethers.BigNumber.from(0);

    try {
      tidBalance = await this.contractGovernance.getCoinBalance(
        checkedsumdaddr,
        token,
      );
    } catch (err) {
      this.logger.error(
        `Failed to get trading account ${checkedsumdaddr} token ${token} balance: ${err}`,
      );
    }
    const amount = {
      balance: ethers.utils.formatUnits(tidBalance, 18),
    };
    this.logger.debug(
      `Trading account ${checkedsumdaddr} token ${token} balance: ${tidBalance}, formated ${amount.balance}`,
    );
    return amount;
  }

  async postTradingAccount(tid: string, operation: string, amount: string) {
    if (!tid) {
      throw new BadRequestException('Bad Request');
    }

    if (!this.commonUtils.isValidEthAddress(tid)) {
      throw new HttpException(
        'Invalid ethereum address',
        HttpStatus.BAD_REQUEST,
      );
    }

    const checkedsumdaddr = ethers.utils.getAddress(tid);
    const bamount = 0 + Number(amount);
    const op = operation.toUpperCase();
    const amountInWei = this.commonUtils.decimalToWei(bamount);

    const signer = new ethers.Wallet(
      ADMIN_PRIVATE_KEY,
      this.commonUtils.provider,
    );

    this.contractGovernance = this.contractGovernance.connect(signer);
    if (op === 'BURN') {
      const tx = await this.contractGovernance.burnCoin(
        checkedsumdaddr,
        'FDE',
        amountInWei,
      );
      await tx.wait(1);
    } else if (op === 'MINT') {
      const tx = await this.contractGovernance.mintCoin(
        checkedsumdaddr,
        'FDE',
        amountInWei,
      );
      await tx.wait(1);
      return;
    } else {
      throw new BadRequestException('Bad Request');
    }
  }

  async getCoinSymbol(tokenaddr: string, tokenindx: BigNumber): Promise<any> {
    let coinsymbol;

    if (!this.commonUtils.isValidEthAddress(tokenaddr)) {
      throw new HttpException(
        'Invalid ethereum address',
        HttpStatus.BAD_REQUEST,
      );
    }

    const checkedsumdaddr = ethers.utils.getAddress(tokenaddr);

    try {
      coinsymbol = await this.contractGovernance.getCoinSymbol(
        checkedsumdaddr,
        tokenindx,
      );
    } catch (err) {
      this.logger.error(
        `Failed to get token ${checkedsumdaddr} symbol: ${err}`,
      );
    }
    this.logger.debug(`Token ${checkedsumdaddr} symbol: ${coinsymbol}`);
    return coinsymbol;
  }
}
