import { Injectable, Logger } from '@nestjs/common';
import { CommonUtils } from '../../utils/common.utils';
import { ethers } from 'ethers';
import * as dotenv from 'dotenv';
import { getContractABI } from '@contracts';
dotenv.config();

const ADMIN_PRIVATE_KEY = process.env.ADMIN_PRIVATE_KEY;

@Injectable()
export class InitializerService {
  private contractPaymentToken: ethers.Contract;
  private contractBourse: ethers.Contract;
  private contractGovernance: ethers.Contract;
  private contractEscrow: ethers.Contract;
  private adminSigner: ethers.Wallet;

  logger = new Logger(InitializerService.name);

  constructor(private commonUtils: CommonUtils) {
    this.adminSigner = new ethers.Wallet(
      ADMIN_PRIVATE_KEY,
      this.commonUtils.provider,
    );
    this.contractPaymentToken = new ethers.Contract(
      this.commonUtils.addressValidatorService.contractAddresses.contractAddressPaymentToken,
      getContractABI('PaymentToken'),
      this.commonUtils.provider,
    );
    this.contractBourse = new ethers.Contract(
      this.commonUtils.addressValidatorService.contractAddresses.contractAddressBourse,
      getContractABI('Bourse'),
      this.commonUtils.provider,
    );
    this.contractGovernance = new ethers.Contract(
      this.commonUtils.addressValidatorService.contractAddresses.contractAddressGovernance,
      getContractABI('Governance'),
      this.commonUtils.provider,
    );
    this.contractEscrow = new ethers.Contract(
      this.commonUtils.addressValidatorService.contractAddresses.contractAddressEscrow,
      getContractABI('Escrow'),
      this.commonUtils.provider,
    );
  }

  async initContracts() {
    this.logger.log(
      `Starting initialization of contracts using admin signer ${this.adminSigner.address}`,
    );

    await this.transferPaymentContractOwnership();
    await this.transferBourseContractOwnership();
    await this.transferEscrowContractOwnership();
    await this.setFameBourseInGovernance();
    await this.registeringFDETokenInGovernance();
    this.logger.log(
      `Initialization of contracts using admin signer ${this.adminSigner.address} was successfully completed!`,
    );
  }

  private async transferPaymentContractOwnership() {
    try {
      const ownerPayment = await this.contractPaymentToken.owner();
      if (ownerPayment != this.contractGovernance.address) {
        const tx = await this.contractPaymentToken
          .connect(this.adminSigner)
          .transferOwnership(this.contractGovernance.address);
        await tx.wait(1);
        this.logger.log(
          `Payment contract ${this.contractPaymentToken.address} ownership was successfully transfered to Governance Contract ${this.contractGovernance.address}!`,
        );
        const ownerPayment = await this.contractPaymentToken.owner();
        this.logger.log(
          `Current payment contract owner is: ${ownerPayment.toString()}`,
        );
      } else {
        this.logger.log(
          `Governance contract is already the owner of Payment Token: ${ownerPayment.toString()}. Now need to set it up`,
        );
      }
    } catch (e) {
      this.logger.error(
        `Error while transfering  payment contract ownership to governance contract`,
        e,
      );
    }
  }

  private async transferBourseContractOwnership() {
    try {
      const ownerBourse = await this.contractBourse.owner();
      if (ownerBourse != this.contractGovernance.address) {
        const tx = await this.contractBourse
          .connect(this.adminSigner)
          .transferOwnership(this.contractGovernance.address);
        await tx.wait(1);
        this.logger.log(
          `Bourse contract ${this.contractBourse.address} ownership transfered to Governance Contract ${this.contractGovernance.address} successfully`,
        );
      } else {
        this.logger.log(
          `Governance Contract ${this.contractGovernance.address} is already the owner of Burse contract ${this.contractBourse.address}. Now need to set it up`,
        );
      }
    } catch (e) {
      this.logger.error(
        `Failed! Bourse contract ${this.contractBourse.address} ownership was not transfered to Governance Contract ${this.contractGovernance.address}`,
        e,
      );
    }
  }

  private async transferEscrowContractOwnership() {
    try {
      const bourseAddress = await this.contractEscrow.getBourse();
      if (bourseAddress == '0x0000000000000000000000000000000000000000') {
        const tx = await this.contractEscrow
          .connect(this.adminSigner)
          .setBourse(this.contractBourse.address);
        await tx.wait(1);
        this.logger.log(
          `Bourse contract ${this.contractBourse.address} set on Escrow contract ${this.contractEscrow.address} successfully!`,
        );
      } else {
        this.logger.log(
          `Escrow ${this.contractEscrow.address} contract has already set up bourse contract ${this.contractBourse.address}`,
        );
      }
    } catch (e) {
      this.logger.error(
        `Failed! Bourse contract ${this.contractBourse.address} was was not set up on Escrow contract`,
        e,
      );
    }
  }

  private async setFameBourseInGovernance() {
    const bourseAddress: string = await this.contractGovernance.famebourse();
    if (bourseAddress === this.contractBourse.address) {
      this.logger.log(
        `Burse contract ${bourseAddress} is already set up in Governance Contract`,
      );
    } else {
      this.logger.log(
        `Setting up Burse contract ${this.contractBourse.address} by setFameBurse method on Governance Contract ${this.contractGovernance.address}`,
      );
      const tx = await this.contractGovernance
        .connect(this.adminSigner)
        .setFameBourse(this.contractBourse.address);
      await tx.wait(1);
      this.logger.log(
        `Fame Burse Contract ${this.contractBourse.address} set in Governance contract ${this.contractGovernance.address} successfully!`,
      );
    }
  }

  private async registeringFDETokenInGovernance() {
    try {
      const tx = await this.contractGovernance
        .connect(this.adminSigner)
        .getCoinAddress('FDE');
      if (tx[0] == this.contractPaymentToken.address && tx[1].toNumber() == 0) {
        this.logger.log(
          `FDE Coin in already set up in Governance contract ${this.contractGovernance.address}`,
        );
      } else {
        this.logger.error(
          `FDE Coin is not properly set up in Governance contract ${this.contractGovernance.address}`,
        );
        this.logger.error(tx);
      }
    } catch (error) {
      if (error.message.includes('Coin does not exist')) {
        this.logger.log(
          `Coin FDE is not visible via Governance Contract ${this.contractGovernance.address}`,
        );
        this.logger.log('Staring registering FDE token in Governance Contract');
        // This symbol will be used for the coin token
        const regixterTx = await this.contractGovernance
          .connect(this.adminSigner)
          .registerCoinToken(this.contractPaymentToken.address, 0, 'FDE');
        await regixterTx.wait(1);
        this.logger.log(
          `FDE coin registered in the governance contract ${this.contractGovernance.address} successfully!`,
        );
      } else {
        this.logger.error(
          `Failed to register FDE coin in Governance contract ${this.contractGovernance.address}`,
          error,
        );
      }
    }
  }
}
