import { ApiProperty, ApiResponse } from '@nestjs/swagger';
import {
  IsNotEmpty,
  IsString,
  IsNumberString,
  MaxLength,
  MinLength,
  IsISO8601,
  ValidateIf,
  isISO8601,
} from 'class-validator';
import { IsOptional } from 'class-validator';
import {
  HasAtLeastOneFieldPresent,
  IsNotEmptyObject,
  IsFutureDate,
} from '../../../validators';
import { Transform } from 'class-transformer';

export class AddOfferDto {
  @ApiProperty({
    description: 'Unique asset ID (Base58Check-encoded SHA3-256 hash).',
  })
  @IsNotEmpty()
  assetid: string;

  @ApiProperty({
    description: 'Unique offering ID (Base58Check-encoded SHA3-256 hash).',
  })
  @IsNotEmpty()
  oid: string;

  @ApiProperty({
    description: 'URL to the Offering Descriptor (downloadable resource).',
  })
  @IsString()
  @IsNotEmpty()
  resource: string;

  @ApiProperty({
    description:
      'Blockchain account designated to receive payments (42-character address).',
  })
  @IsNotEmpty()
  @MaxLength(42)
  @MinLength(42)
  beneficiary: string;

  @ApiProperty({
    description:
      'Amount of FDE currency requested by the seller (decimal number).',
  })
  @IsNotEmpty()
  @IsNumberString()
  price: number;

  @ApiProperty({ description: 'Access duration in seconds.' })
  @IsOptional()
  @Transform(({ value }) => {
    if (isISO8601(value)) {
      return Math.floor(new Date(value).getTime() / 1000).toString();
    }
    if (!isNaN(Number(value))) {
      return value; // Already a valid timestamp string
    }
    throw new Error(
      'Invalid date format. Must be ISO8601 string or Unix timestamp string.',
    );
  })
  cap_duration: string;

  @ApiProperty({
    description: 'Allowed number of downloads (positive integer).',
  })
  @IsOptional()
  @IsNumberString()
  cap_downloads: string;

  @ApiProperty({
    description:
      'Download allowance in bytes (e.g., nKB, nMB, nGB, nTB, nPB, nEB).',
  })
  @IsOptional()
  cap_volume: string;

  @HasAtLeastOneFieldPresent(['cap_volume', 'cap_downloads', 'cap_duration'], {
    message:
      'At least one of cap_volume, cap_downloads, or cap_duration must be provided.',
  })
  _cap: any; // Dummy field to enforce validation

  @ApiProperty({
    description:
      'Download allowance in bytes (e.g., nKB, nMB, nGB, nTB, nPB, nEB).',
  })
  @IsNotEmptyObject()
  cds_target: Record<string, any>;

  @ApiProperty({ description: 'Service level specification (JSON format).' })
  @IsOptional()
  cds_sl: Record<string, any>;
}

export class OfferintIdDto {
  @ApiProperty({
    description:
      'unique ID of offering as it was used during the create an offer call - provided by the user',
  })
  @IsNotEmpty()
  oid: string;
}

export class FullOfferInfoDto {
  @ApiProperty({
    description:
      'Amount of FDE currency requested by the seller (decimal number as string).',
  })
  @IsNotEmpty()
  @IsString()
  dataAccessPrice: string;

  @ApiProperty({
    description: 'Access duration in seconds (as string).',
  })
  @IsNotEmpty()
  @IsString()
  capDuration: string;

  @ApiProperty({
    description: 'Allowed number of downloads (positive integer).',
  })
  @IsOptional()
  capDownloads: string;

  @ApiProperty({
    description:
      'Download allowance in bytes (e.g., nKB, nMB, nGB, nTB, nPB, nEB).',
  })
  @IsOptional()
  capVolume: string;

  @ApiProperty({
    description: 'CDS target specification (JSON format).',
    type: 'object',
  })
  @IsNotEmptyObject()
  cdsTarget: Record<string, any>;

  @ApiProperty({
    description: 'Service level specification (JSON format).',
    type: 'object',
  })
  @IsOptional()
  cdsSl: Record<string, any>;

  @ApiProperty({
    description: 'Unique offering ID (Base58Check-encoded SHA3-256 hash).',
  })
  @IsNotEmpty()
  @IsString()
  oid: string;

  @ApiProperty({
    description: 'Data provider address (blockchain account address).',
  })
  @IsNotEmpty()
  @IsString()
  dataProvider: string;
}
