import { Injectable, Logger } from '@nestjs/common';
import { CommonUtils, Contracts } from '../../utils/common.utils';
import { PopulatedTransaction, ethers, BigNumber } from 'ethers';
import axios from 'axios';
import { HttpException, HttpStatus } from '@nestjs/common';
import { FullOfferInfoDto } from './offering-dto/offering.dto';
import * as dotenv from 'dotenv';
dotenv.config();

const ADMIN_PRIVATE_KEY = process.env.ADMIN_PRIVATE_KEY;
const PT_CONFIRM_OFFERING_ENDPOINT = process.env.PT_CONFIRM_OFFERING_ENDPOINT;

@Injectable()
export class OfferingService {
  private logger = new Logger(OfferingService.name);
  private contracts: Contracts;

  constructor(private commonUtils: CommonUtils) {
    this.contracts = this.commonUtils.getContracts();
  }

  async addAssetUnsigned(
    assetid: string,
    oid: string,
    resource: string,
    beneficiary: string,
    price: number,
    cap_duration: string,
    cap_downloads: string,
    cap_volume: string,
    cds_target: any,
    cds_sl: any,
  ): Promise<PopulatedTransaction> {
    const priceInWei: ethers.BigNumber = this.commonUtils.decimalToWei(price);

    // creating custom data to log to transaction
    const data: any = {
      assetid: assetid,
      oid: oid,
      resource: resource,
      beneficiary: beneficiary,
      price: priceInWei,
      cap_duration: cap_duration || '',
      cap_downloads: cap_downloads || '',
      cap_volume: cap_volume || '',
      cds_target: cds_target || '',
      cds_sl: cds_sl || '',
    };

    const jsonData: string = JSON.stringify(data);
    const dataAsBytes: string = ethers.utils.hexlify(
      ethers.utils.toUtf8Bytes(jsonData),
    );

    const ownerSigner = new ethers.Wallet(
      ADMIN_PRIVATE_KEY,
      this.commonUtils.provider,
    );

    // The following code is removed after cap_duration in seconds have been introduced
    //const currentDateTime = new Date();
    //let timeDifferenceInMinutes: number;
    //if (cap_expires) {
    //  const expirationDateTime = new Date(cap_expires);
    //  timeDifferenceInMinutes = Math.round(
    //    (expirationDateTime.getTime() - currentDateTime.getTime()) /
    //      (1000 * 60),
    //  );
    //  console.log('exp time: ' + timeDifferenceInMinutes);
    //}

    const cdsTargetJson = JSON.stringify(cds_target);
    const cdsSlJson = JSON.stringify(cds_sl);

    try {
      const unsignedTx = await this.contracts.contractOfferingToken
        .connect(ownerSigner)
        .populateTransaction.addAsset(
          assetid,
          oid,
          resource,
          beneficiary,
          priceInWei,
          cap_duration || 0,
          cap_downloads || '',
          cap_volume || '',
          cdsTargetJson || '',
          cdsSlJson || '',
          dataAsBytes,
        );
      return unsignedTx;
    } catch (err) {
      return err;
    }
  }

  async addAssetSigned(
    assetid: string,
    oid: string,
    resource: string,
    beneficiary: string,
    price: number,
    cap_duration: string,
    cap_downloads: string,
    cap_volume: string,
    cds_target: any,
    cds_sl: any,
  ): Promise<string> {
    const priceInWei: ethers.BigNumber = this.commonUtils.decimalToWei(price);

    // creating custom data to log to transaction
    const data: any = {
      assetid: assetid,
      oid: oid,
      resource: resource,
      beneficiary: beneficiary,
      price: priceInWei,
      cap_duration: cap_duration || '',
      cap_downloads: cap_downloads || '',
      cap_volume: cap_volume || '',
      cds_target: cds_target || '',
      cds_sl: cds_sl || '',
    };

    const exists = await this.contracts.contractOfferingToken.tokenExists(oid);
    if (exists) {
      throw new HttpException(
        `Asset with oid ${oid} already exists`,
        HttpStatus.CONFLICT,
      );
    }

    const jsonData: string = JSON.stringify(data);
    const dataAsBytes: string = ethers.utils.hexlify(
      ethers.utils.toUtf8Bytes(jsonData),
    );

    const ownerSigner = new ethers.Wallet(
      ADMIN_PRIVATE_KEY,
      this.commonUtils.provider,
    );

    // The following code was removed after cap_duration in seconds was used.
    //const currentDateTime = new Date();
    //let timeDifferenceInMinutes: number;
    //if (cap_expires) {
    //  const expirationDateTime = new Date(cap_expires);
    //  timeDifferenceInMinutes = Math.round(
    //    (expirationDateTime.getTime() - currentDateTime.getTime()) /
    //      (1000 * 60),
    //  );
    // }

    const cdsTargetJson = JSON.stringify(cds_target);
    const cdsSlJson = JSON.stringify(cds_sl);

    try {
      const tx = await this.contracts.contractOfferingToken
        .connect(ownerSigner)
        .addAsset(
          assetid,
          oid,
          resource,
          beneficiary,
          priceInWei,
          cap_duration || 0,
          cap_downloads || '',
          cap_volume || '',
          cdsTargetJson || '',
          cdsSlJson || '',
          dataAsBytes,
        );
      const receipt = await tx.wait();
      this.informPTModule(oid);
      return receipt.transactionHash;
    } catch (err) {
      return err;
    }
  }

  async burnAsset(oid: string): Promise<string> {
    // creating custom data to log to transaction
    const ownerSigner = new ethers.Wallet(
      ADMIN_PRIVATE_KEY,
      this.commonUtils.provider,
    );

    const exists = await this.contracts.contractOfferingToken.tokenExists(oid);
    if (!exists) {
      throw new HttpException(
        `Asset with oid ${oid} does not exists`,
        HttpStatus.NOT_FOUND,
      );
    }

    const tx = await this.contracts.contractOfferingToken
      .connect(ownerSigner)
      .removeOffering(oid);

    const receipt = await tx.wait();

    return receipt.transactionHash;
  }

  async informPTModule(oid: string): Promise<void> {
    const pathURL = `${PT_CONFIRM_OFFERING_ENDPOINT}/${oid}/confirm`;

    try {
      await axios.put(pathURL);
      this.logger.log(`Confirm offering calling put ${pathURL}`);
    } catch (error) {
      this.logger.error(
        `Failed to confirm offering ${oid} by calling ${pathURL}`,
      );
      this.logger.error(error);
    }
  }

  async getAssetExistance(oid: string): Promise<boolean> {
    const exists = await this.contracts.contractOfferingToken.tokenExists(oid);

    return exists;
  }

  async getAssetHistoricalExistance(oid: string): Promise<boolean> {
    const exists =
      await this.contracts.contractOfferingToken.historicallyTokenExists(oid);

    return exists;
  }

  async getTokenURI(oid: string): Promise<string> {
    const uri = await this.contracts.contractOfferingToken.tokenURIHash(oid);

    return uri;
  }

  async getOfferIdInfo(oid: string): Promise<any> {
    const OfferInfo =
      await this.contracts.contractOfferingToken.getOfferIdInfo(oid);
    return OfferInfo;
  }
  async getFullOfferIdInfo(oid: string): Promise<FullOfferInfoDto> {
    try {
      const offerInfo = await this.getOfferIdInfo(oid);

      return {
        dataAccessPrice: offerInfo[0].toString(),
        capDuration: offerInfo[1].toString(),
        capDownloads: offerInfo[2],
        capVolume: offerInfo[3],
        cdsTarget: offerInfo[4],
        cdsSl: offerInfo[5],
        oid: offerInfo[6],
        dataProvider: offerInfo[7],
      };
    } catch (error) {
      throw new HttpException(error, HttpStatus.NOT_FOUND);
    }
  }

  async getDataAccessTokenInfo(tokenId: string): Promise<any> {
    const oid =
      await this.contracts.contractOfferingToken.getOidFromHash(tokenId);
    const OfferInfo =
      await this.contracts.contractOfferingToken.getOfferIdInfo(oid);
    return OfferInfo;
  }

  async getMintedTokens(): Promise<number[]> {
    const mintedTokensData =
      await this.contracts.contractOfferingToken.getMintedTokens();
    const mintedTokens: number[] = mintedTokensData.map((token) =>
      parseInt(token._hex, 36),
    );

    return mintedTokens;
  }

  // This function can be used to take hash of oid or assetid
  // note that ERC721 offering token index is equal to oidHash.
  async getIDHash(oid: string): Promise<any> {
    const uri = await this.contracts.contractOfferingToken.getIDHash(oid);
    return uri;
  }

  // Note that ERC721 offering token index is equal to oidHash
  async getAssetidHashOfOidHash(oidHash: BigNumber): Promise<BigNumber> {
    const assetidHash =
      await this.contracts.contractOfferingToken.getAssetidHashOfOidHash(
        oidHash,
      );

    return assetidHash;
  }

  async getAssetidOfOid(oid: string): Promise<string> {
    const assetid =
      await this.contracts.contractOfferingToken.getAssetidOfOid(oid);
    return assetid;
  }

  async getAssetidFromHash(assetidHash: BigNumber): Promise<string> {
    const assetid =
      await this.contracts.contractOfferingToken.getAssetidFromHash(
        assetidHash,
      );
    return assetid;
  }

  async getOidFromHash(oidHash: BigNumber): Promise<string> {
    const oid =
      await this.contracts.contractOfferingToken.getOidFromHash(oidHash);
    return oid;
  }
}
