import { ApiProperty } from '@nestjs/swagger';
import {
  IsEthereumAddress,
  IsNotEmpty,
  IsNumberString,
  IsString,
  MaxLength,
  MinLength,
} from 'class-validator';

export class MintPaymentTokens {
  @ApiProperty()
  @IsNotEmpty()
  @IsEthereumAddress()
  userAddress: string;

  @ApiProperty()
  @IsNotEmpty()
  @IsNumberString()
  @MaxLength(20)
  amount: number;
}

export class GetBalanceDto {
  @ApiProperty()
  @IsNotEmpty()
  @IsEthereumAddress()
  userAddress: string;
}

export class GetBalanceResponseDto {
  @ApiProperty()
  @IsNotEmpty()
  @IsNumberString()
  balance: number;
}

export class ApproveToSpendPaymentTokensDto {
  @ApiProperty({
    description:
      'Private key of the user who is approving the operator (Burse Contract) to spend tokens on his behalf.',
  })
  @IsNotEmpty()
  @IsString()
  privateKey: string;

  @ApiProperty({
    description:
      'Amount of FDE tokens in decimal number to approve for spending',
  })
  @IsNotEmpty()
  @IsNumberString()
  @MaxLength(20)
  approveAmount: number;
}

export class GetPaymentAllowanceOnOperatorDto {
  @ApiProperty({
    description:
      'Ethereum address of the user whose allowance is being checked.',
  })
  @IsNotEmpty()
  @IsEthereumAddress()
  userAddress: string;
}

export class TransferPaymentTokensDto {
  @ApiProperty()
  @IsNotEmpty()
  @MaxLength(64)
  @MinLength(64)
  privateKey: string;

  @ApiProperty()
  @IsNotEmpty()
  @IsEthereumAddress()
  transferToAddress: string;

  @ApiProperty()
  @IsNotEmpty()
  @IsNumberString()
  @MaxLength(20)
  tokenTransferAmount: number;
}

export class BurnPaymentTokenDto {
  @ApiProperty()
  @IsNotEmpty()
  @IsEthereumAddress()
  userAddress: string;

  @ApiProperty()
  @IsNotEmpty()
  @IsEthereumAddress()
  burnTokenAddress: string;

  @ApiProperty()
  @IsNotEmpty()
  @IsNumberString()
  @MaxLength(20)
  burnAmount: number;
}

export class IncreaseAllowanceDto {
  @ApiProperty({
    description: 'Amount of tokens to approve in human units',
  })
  @IsNumberString()
  amountToAllow: string;

  @ApiProperty({
    description: 'Ethereum address to approve for token spending',
    example: '0xAbC123...DEF456',
  })
  @IsEthereumAddress()
  toApproveAddress: string;
}

export class PaymentAllowanceQueryDto {
  @ApiProperty({
    description: "User's Ethereum address holding the tokens",
    example: '0x123AbC...456DEF',
  })
  @IsEthereumAddress()
  userAddress: string;

  @ApiProperty({
    description: 'Ethereum address to check allowance for',
    example: '0x789XYZ...123ABC',
  })
  @IsEthereumAddress()
  allowanceCheckAddress: string;
}
