import { Injectable, Logger } from '@nestjs/common';
import { CommonUtils, Contracts } from '../../utils/common.utils';
import { ethers } from 'ethers';
import * as dotenv from 'dotenv';
dotenv.config();

const ADMIN_PRIVATE_KEY = process.env.ADMIN_PRIVATE_KEY;

@Injectable()
export class PaymentService {
  private logger = new Logger(PaymentService.name);
  private contracts: Contracts;

  constructor(private commonUtils: CommonUtils) {
    this.contracts = this.commonUtils.getContracts();
  }

  async mintPaymentTokens(
    userAddress: string,
    amount: number,
  ): Promise<string> {
    const amountInWei = this.commonUtils.decimalToWei(amount);

    const data = {
      mintTo: userAddress,
      mintAmount: amountInWei,
    };

    const jsonData: string = JSON.stringify(data);
    const dataAsBytes: string = ethers.utils.hexlify(
      ethers.utils.toUtf8Bytes(jsonData),
    );

    const ownerSigner = new ethers.Wallet(
      ADMIN_PRIVATE_KEY,
      this.commonUtils.provider,
    );

    const tx = await this.contracts.contractPaymentToken
      .connect(ownerSigner)
      .mint(userAddress, amountInWei, dataAsBytes);

    const receipt = await tx.wait(1);
    return receipt.transactionHash;
  }

  async getBalanceOfPaymentToken(userAddress: string): Promise<string> {
    const balance: ethers.BigNumber =
      await this.contracts.contractPaymentToken.balanceOf(userAddress);

    return balance.toString();
  }

  async approveToSpendPaymentToken(
    privateKey: string,
    approveAmount: number,
    toApproveAddress = this.contracts.bourseContract.address,
  ): Promise<string> {
    const amountToApproveInWei: ethers.BigNumber =
      this.commonUtils.decimalToWei(approveAmount);
    const signer = new ethers.Wallet(privateKey, this.commonUtils.provider);
    // creating, signing and sending the transaction
    const tx = await this.contracts.contractPaymentToken
      .connect(signer)
      .approve(toApproveAddress, amountToApproveInWei);
    const receipt = await tx.wait();
    this.logger.log(
      `Approval for spending ${approveAmount} amount of Payment on ${toApproveAddress} tokens succeeded! Transaction hash: ` +
        receipt.transactionHash,
    );
    return receipt.transactionHash;
  }

  async approveToSpendUnlimitedPaymentToken(
    toApproveAddress = this.contracts.bourseContract.address,
  ) {
    try {
      const unsignedTx =
        await this.contracts.contractPaymentToken.populateTransaction.approve(
          toApproveAddress,
          ethers.constants.MaxUint256,
        );

      this.logger.log(
        'Returning unsigned transcation allowing to spend unlimited amount of FDE tokes: ' +
          JSON.stringify(unsignedTx),
      );
      return unsignedTx;
    } catch (error) {
      this.logger.error('Failed to prepare the transaction:', error);
      throw error;
    }
  }

  async getIncreaseAllowanceTx(
    toApproveAddress: string,
    amountToAllow: string,
  ) {
    const amount = ethers.utils.parseUnits(amountToAllow, 18);
    try {
      const unsignedTx =
        await this.contracts.contractPaymentToken.populateTransaction.increaseAllowance(
          toApproveAddress,
          amount,
        );

      this.logger.log(
        'Returning unsigned transaction increasing allowance to spend FDE tokens: ' +
          JSON.stringify(unsignedTx),
      );
      return unsignedTx;
    } catch (error) {
      this.logger.error('Failed to prepare the transaction:', error);
      throw error;
    }
  }

  async getPaymentTokenAllowance(
    userAddress: string,
    allowedAddress: string,
  ): Promise<number> {
    const allowanceAmount: ethers.BigNumber =
      await this.contracts.contractPaymentToken.allowance(
        userAddress,
        allowedAddress,
      );

    return this.commonUtils.weiToDecimals(allowanceAmount);
  }

  async getPaymentTokenAllowanceOnOperator(
    userAddress: string,
    allowanceCheckAddress = this.contracts.bourseContract.address,
  ): Promise<number> {
    const allowanceAmount: ethers.BigNumber =
      await this.contracts.contractPaymentToken.allowance(
        userAddress,
        allowanceCheckAddress,
      );

    return this.commonUtils.weiToDecimals(allowanceAmount);
  }

  async transferPaymentTokens(
    privateKey: string,
    transferToAddress: string,
    tokenTransferAmount: number,
  ): Promise<string> {
    const amountInWei: ethers.BigNumber =
      this.commonUtils.decimalToWei(tokenTransferAmount);

    const signer = new ethers.Wallet(privateKey, this.commonUtils.provider);

    // creating, signing and sending the transaction
    const tx = await this.contracts.contractPaymentToken
      .connect(signer)
      .transfer(transferToAddress, amountInWei);

    const receipt = await tx.wait();

    return receipt.transactionHash;
  }

  async burnPaymentToken(
    userAddress: string,
    burnTokenAddress: string,
    burnAmount: number,
  ): Promise<string> {
    const amountToBurnInWei: ethers.BigNumber =
      this.commonUtils.decimalToWei(burnAmount);

    // creating custom data to log to transaction
    const data = {
      owner: userAddress,
      burnFrom: burnTokenAddress,
      amountBurned: amountToBurnInWei,
    };

    const jsonData: string = JSON.stringify(data);
    const dataAsBytes: string = ethers.utils.hexlify(
      ethers.utils.toUtf8Bytes(jsonData),
    );

    const ownerSigner = new ethers.Wallet(
      ADMIN_PRIVATE_KEY,
      this.commonUtils.provider,
    );

    const tx = await this.contracts.contractPaymentToken
      .connect(ownerSigner)
      .burn(burnTokenAddress, amountToBurnInWei, dataAsBytes);

    const receipt = await tx.wait();

    return receipt.transactionHash;
  }
}
