import { Injectable, Logger } from '@nestjs/common';
import { CommonUtils, Contracts } from '../../utils/common.utils';
import { OfferingService } from '../offering/offering.service';
import * as dotenv from 'dotenv';
dotenv.config();

@Injectable()
export class PurchasesService {
  private contracts: Contracts;

  logger = new Logger(PurchasesService.name);

  constructor(
    private commonUtils: CommonUtils,
    private offeringService: OfferingService,
  ) {
    this.contracts = this.commonUtils.getContracts();
  }

  async getUserDataAccessOfferIds(
    userAddress: string,
    tokenType?: 'SUB' | 'PAYG' | 'PAYU',
  ): Promise<Record<string, string[]>> {
    const processHashesToOids = async (hashes: any[]): Promise<string[]> => {
      const results: string[] = [];

      // Process sequentially with a small delay between each call
      for (const hash of hashes) {
        const result = await this.offeringService.getOidFromHash(hash);
        results.push(result);

        // Small delay to avoid overwhelming the blockchain
        await new Promise((resolve) => setTimeout(resolve, 200));
      }

      return results;
    };

    if (tokenType) {
      let hashes: any[];

      if (tokenType === 'SUB') {
        hashes =
          await this.contracts.contractTradingManagementExecutor.getAllUserDataAccessSubscriptionOfferIds(
            userAddress,
          );
      } else if (tokenType === 'PAYG') {
        hashes =
          await this.contracts.contractTradingManagementExecutor.getAllUserDataAccessPAYGOfferIds(
            userAddress,
          );
      } else {
        // tokenType is 'PAYU'
        hashes =
          await this.contracts.contractTradingManagementExecutor.getAllUserDataAccessPAYUOfferIds(
            userAddress,
          );
      }

      const offerIds = await processHashesToOids(hashes);
      return { [tokenType]: offerIds };
    }

    const [subHashes, paygHashes, payuHashes] = await Promise.all([
      this.contracts.contractTradingManagementExecutor.getAllUserDataAccessSubscriptionOfferIds(
        userAddress,
      ),
      this.contracts.contractTradingManagementExecutor.getAllUserDataAccessPAYGOfferIds(
        userAddress,
      ),
      this.contracts.contractTradingManagementExecutor.getAllUserDataAccessPAYUOfferIds(
        userAddress,
      ),
    ]);

    const [subOids, paygOids, payuOids] = await Promise.all([
      processHashesToOids(subHashes),
      processHashesToOids(paygHashes),
      processHashesToOids(payuHashes),
    ]);

    return {
      SUB: subOids,
      PAYG: paygOids,
      PAYU: payuOids,
    };
  }
  async getUserDataAccessTokenIds(
    userAddress: string,
    tokenType?: 'SUB' | 'PAYG' | 'PAYU',
  ): Promise<Record<string, string[]>> {
    if (tokenType) {
      let tokenIdsAsBigNumbers;

      if (tokenType === 'SUB') {
        tokenIdsAsBigNumbers =
          await this.contracts.contractTradingManagementExecutor.getAllUserDataAccessSubscriptionOfferIds(
            userAddress,
          );
      } else if (tokenType === 'PAYG') {
        tokenIdsAsBigNumbers =
          await this.contracts.contractTradingManagementExecutor.getAllUserDataAccessPAYGOfferIds(
            userAddress,
          );
      } else {
        // tokenType is 'PAYU'
        tokenIdsAsBigNumbers =
          await this.contracts.contractTradingManagementExecutor.getAllUserDataAccessPAYUOfferIds(
            userAddress,
          );
      }

      const tokenIds = tokenIdsAsBigNumbers.map((bn) => bn.toString());
      return { [tokenType]: tokenIds };
    }

    const [subTokenHashes, paygTokenHashes, payuTokenHashes] =
      await Promise.all([
        this.contracts.contractTradingManagementExecutor.getAllUserDataAccessSubscriptionOfferIds(
          userAddress,
        ),
        this.contracts.contractTradingManagementExecutor.getAllUserDataAccessPAYGOfferIds(
          userAddress,
        ),
        this.contracts.contractTradingManagementExecutor.getAllUserDataAccessPAYUOfferIds(
          userAddress,
        ),
      ]);

    return {
      SUB: subTokenHashes.map((bn) => bn.toString()),
      PAYG: paygTokenHashes.map((bn) => bn.toString()),
      PAYU: payuTokenHashes.map((bn) => bn.toString()),
    };
  }
}
