import {
  Injectable,
  BadRequestException,
  InternalServerErrorException,
  HttpException,
  HttpStatus,
  Logger,
} from '@nestjs/common';
import { ethers, BigNumber } from 'ethers';
import * as dotenv from 'dotenv';
import { CommonUtils, Contracts } from '../../utils/common.utils';
import { GovernanceService } from '../governance/governance.service';
import { OfferingService } from '../offering/offering.service';

dotenv.config();

// const ChainProviderURL = `http://${process.env.BESU_RPC}` ;

// Note in this file ([ta1,ti1],[ta2,ti2])  is the token pair where
// ta1 : first token contract address in the token pair
// ti1 : first token index in the token pair
// ta2 : second token contract address in the token pair
// ti2 : second token index in the token pair
// for ERC20 token contract, index is taken to be 0.

type SellData = {
  ta1: string;
  ti1: BigNumber;
  ta2: string;
  ti2: BigNumber;
  saleTxNo: BigNumber;
};

type SellResult = [
  orderno: number,
  price: BigNumber,
  amount: BigNumber,
  unixtime: number,
];

@Injectable()
export class TradingHistoryService {
  private contracts: Contracts;

  logger = new Logger(TradingHistoryService.name);

  constructor(
    private commonUtils: CommonUtils,
    private offeringService: OfferingService,
    private governanceService: GovernanceService,
  ) {
    this.contracts = this.commonUtils.getContracts();
  }

  async getNumberOfTradePairs(): Promise<any> {
    let retNum;

    try {
      const numTradePairs =
        await this.contracts.bourseContract.getNumberOfTradePairs();
      retNum = Number(numTradePairs);
    } catch (err) {
      console.error(err);
      throw new InternalServerErrorException('Internal Service Error');
    }

    return retNum;
  }

  async getIthTradePair(ithp: number): Promise<any> {
    let retValues = [];

    try {
      const ithTradePair =
        await this.contracts.bourseContract.getIthTradePair(ithp);
      retValues = [
        ithTradePair[0],
        ithTradePair[1],
        ithTradePair[2],
        ithTradePair[3],
      ];
    } catch (err) {
      console.error(err);
      throw new InternalServerErrorException('Internal Service Error');
    }

    return retValues;
  }

  async getTotalNumVolBuys(
    ta1: string, // ta1 : first token contract address in the pair
    ti1: BigNumber,
    ta2: string,
    ti2: BigNumber,
  ): Promise<any> {
    let retValues = [];

    try {
      const ta1Checked = ethers.utils.getAddress(ta1);
      const ta2Checked = ethers.utils.getAddress(ta2);
      const totnumvol =
        await this.contracts.bourseContract.getTotalNumVolBuyBids(
          ta1Checked,
          ti1,
          ta2Checked,
          ti2,
        );
      retValues = [BigNumber.from(totnumvol[0]), BigNumber.from(totnumvol[1])];
    } catch (err) {
      console.error(err);
      throw new InternalServerErrorException('Internal Service Error');
    }

    return retValues;
  }

  async getTotalNumVolSells(
    ta1: string, // ta1 : first token contract address in the pair
    ti1: BigNumber, // ti1 : first token index in the pair
    ta2: string, // ta2 : second token contract address in the pair
    ti2: BigNumber, // ti2 : first token index in the pair
  ): Promise<any> {
    let retValues = [];

    try {
      const ta1Checked = ethers.utils.getAddress(ta1);
      const ta2Checked = ethers.utils.getAddress(ta2);
      const totalNumVol =
        await this.contracts.bourseContract.getTotalNumVolSellBids(
          ta1Checked,
          ti1,
          ta2Checked,
          ti2,
        );
      retValues = [
        BigNumber.from(totalNumVol[0]),
        BigNumber.from(totalNumVol[1]),
      ];
    } catch (err) {
      console.error(err);
      throw new InternalServerErrorException('Internal Service Error');
    }

    return retValues;
  }

  async getNumberOfTraderBuys(
    traderAddr: string, // trader address
    ta1: string, // ta1 : first token contract address in the token pair
    ti1: BigNumber, // ti1 : first token index in the token pair
    ta2: string, // ta2 : second token contract address in the token pair
    ti2: BigNumber, // ti2 : first token index in the token pair
  ): Promise<any> {
    let numTraderBuys;

    try {
      const traderAddrChecked = ethers.utils.getAddress(traderAddr);
      const ta1Checked = ethers.utils.getAddress(ta1);
      const ta2Checked = ethers.utils.getAddress(ta2);
      numTraderBuys =
        await this.contracts.bourseContract.getNumberOfTraderBuyBids(
          traderAddrChecked,
          ta1Checked,
          ti1,
          ta2Checked,
          ti2,
        );
    } catch (err) {
      console.error(err);
      throw new InternalServerErrorException('Internal Service Error');
    }

    return Number(numTraderBuys);
  }

  async getNumberOfTraderSells(
    traderAddr: string, // trader address
    ta1: string, // ta1 : first token contract address in the pair
    ti1: BigNumber, // ti1 : first token index in the pair
    ta2: string,
    ti2: BigNumber,
  ): Promise<any> {
    let numTraderSells;

    try {
      const traderAddrChecked = ethers.utils.getAddress(traderAddr);
      const ta1Checked = ethers.utils.getAddress(ta1);
      const ta2Checked = ethers.utils.getAddress(ta2);
      numTraderSells =
        await this.contracts.bourseContract.getNumberOfTraderSellBids(
          traderAddrChecked,
          ta1Checked,
          ti1,
          ta2Checked,
          ti2,
        );
    } catch (err) {
      console.error(err);
      throw new InternalServerErrorException('Internal Service Error');
    }

    return Number(numTraderSells);
  }

  async getIthTraderSell(
    ith: number, //  i'th iteration  i=1,2, etc.
    traderAddr: string, // trader ethereum address
    ta1: string, // ta1 : first token contract address in the pair
    ti1: BigNumber, // ti1 : first token index in the pair
    ta2: string,
    ti2: BigNumber,
  ): Promise<any> {
    let retValues = [];

    try {
      const traderAddrChecked = ethers.utils.getAddress(traderAddr);
      const ta1Checked = ethers.utils.getAddress(ta1);
      const ta2Checked = ethers.utils.getAddress(ta2);
      const ithSell = await this.contracts.bourseContract.getIthTraderSellBid(
        ith,
        traderAddrChecked,
        ta1Checked,
        ti1,
        ta2Checked,
        ti2,
      );
      retValues = [
        Number(ithSell[0]),
        Number(ithSell[1]),
        Number(ithSell[2]),
        Number(ithSell[3]),
        Number(ithSell[4]),
      ];
    } catch (err) {
      console.error(err);
      throw new InternalServerErrorException('Internal Service Error');
    }

    return retValues;
  }

  async getIthTraderBuy(
    ith: number, //  i'th iteration  i=1,2, etc.
    traderAddr: string,
    ta1: string, // ta1 : first token contract address in the pair
    ti1: BigNumber, // ti1 : first token index in the pair
    ta2: string,
    ti2: BigNumber,
  ): Promise<any> {
    let retValues = [];
    try {
      const traderAddrChecked = ethers.utils.getAddress(traderAddr);
      const ta1Checked = ethers.utils.getAddress(ta1);
      const ta2Checked = ethers.utils.getAddress(ta2);
      const ithSell = await this.contracts.bourseContract.getIthTraderBuyBid(
        ith,
        traderAddrChecked,
        ta1Checked,
        ti1,
        ta2Checked,
        ti2,
      );
      retValues = [
        Number(ithSell[0]),
        Number(ithSell[1]),
        Number(ithSell[2]),
        Number(ithSell[3]),
        Number(ithSell[4]),
      ];
    } catch (err) {
      console.error(err);
      throw new InternalServerErrorException('Internal Service Error');
    }

    return retValues;
  }

  async getNumberOfSells(
    ta1: string, // ta1 : first token contract address in the pair
    ti1: BigNumber, // ti1 : first token index in the pair
    ta2: string,
    ti2: BigNumber,
  ): Promise<any> {
    let numsells: BigNumber;

    try {
      const ta1Checked = ethers.utils.getAddress(ta1);
      const ta2Checked = ethers.utils.getAddress(ta2);
      numsells = await this.contracts.bourseContract.getNumberOfSellBids(
        ta1Checked,
        ti1,
        ta2Checked,
        ti2,
      );
    } catch (err) {
      console.error(err);
      throw new InternalServerErrorException('Internal Service Error');
    }

    return numsells.toNumber();
  }

  async getNumberOfBuys(
    ta1: string, // ta1 : first token contract address in the pair
    ti1: BigNumber, // ti1 : first token index in the pair
    ta2: string,
    ti2: BigNumber,
  ): Promise<any> {
    let numbuys;
    try {
      const ta1Checked = ethers.utils.getAddress(ta1);
      const ta2Checked = ethers.utils.getAddress(ta2);
      numbuys = await this.contracts.bourseContract.getNumberOfBuyBids(
        ta1Checked,
        ti1,
        ta2Checked,
        ti2,
      );
    } catch (err) {
      console.error(err);
      throw new InternalServerErrorException('Internal Service Error');
    }

    return Number(numbuys);
  }

  async getIthBuy(
    ith: number, //  i'th iteration  i=1,2, etc.
    ta1: string, // ta1 : first token contract address in the pair
    ti1: BigNumber, // ti1 : first token index in the pair
    ta2: string,
    ti2: BigNumber,
  ): Promise<any> {
    let retValues = [];
    try {
      const ta1Checked = ethers.utils.getAddress(ta1);
      const ta2Checked = ethers.utils.getAddress(ta2);
      const ithBuy = await this.contracts.bourseContract.getIthBuyBid(
        ith,
        ta1Checked,
        ti1,
        ta2Checked,
        ti2,
      );
      retValues = [
        Number(ithBuy[0]),
        Number(ithBuy[1]),
        Number(ithBuy[2]),
        Number(ithBuy[3]),
      ];
    } catch (err) {
      console.error(err);
      throw new InternalServerErrorException('Internal Service Error');
    }

    return retValues;
  }

  async getIthSell(
    ith: number, //  i'th iteration  i=1,2, etc.
    ta1: string, // ta1 : first token contract address in the pair
    ti1: BigNumber, // ti1 : first token index in the pair
    ta2: string,
    ti2: BigNumber,
  ): Promise<any> {
    let retValues = [];
    try {
      const ta1Checked = ethers.utils.getAddress(ta1);
      const ta2Checked = ethers.utils.getAddress(ta2);
      const ithSell = await this.contracts.bourseContract.getIthSellBid(
        ith,
        ta1Checked,
        ti1,
        ta2Checked,
        ti2,
      );
      retValues = [
        Number(ithSell[0]),
        Number(ithSell[1]),
        Number(ithSell[2]),
        Number(ithSell[3]),
      ];
    } catch (err) {
      console.error(err);
      throw new InternalServerErrorException('Internal Service Error');
    }

    return retValues;
  }

  async getAssetOrOIdTradeHistory(
    idAssetOrOId: string,
    flagOId: boolean,
  ): Promise<any> {
    try {
      let assetIdHash;
      let oidHash;
      let assetidStr;
      let oidStr;

      if (flagOId) {
        oidHash = await this.offeringService.getIDHash(idAssetOrOId);
        oidStr = idAssetOrOId;
      } else {
        assetIdHash = await this.offeringService.getIDHash(idAssetOrOId);
        assetidStr = idAssetOrOId;
      }

      const retNumTradePairs = await this.getNumberOfTradePairs();
      const allRetValues = [];
      for (let itp = 0; itp < retNumTradePairs; itp++) {
        // tokenPair[0] : first token contract address in the token pair
        // tokenPair[1] : first token index in the pair
        // tokenPair[2] : first token contract address in the token pair
        // tokenPair[3] : first token index in the token pair
        const tokenPair = await this.getIthTradePair(itp);
        const numSells = await this.getNumberOfSells(
          tokenPair[0],
          tokenPair[1],
          tokenPair[2],
          tokenPair[3],
        );
        const coinSymbol = await this.governanceService.getCoinSymbol(
          tokenPair[2],
          tokenPair[3],
        );
        let outputTrade = false;
        const tokIndx = tokenPair[1];
        const tokAssetId =
          await this.offeringService.getAssetidHashOfOidHash(tokIndx);
        if (flagOId) {
          outputTrade = oidHash.eq(tokIndx);
          assetidStr =
            await this.offeringService.getAssetidFromHash(tokAssetId);
        } else {
          outputTrade = assetIdHash.eq(tokAssetId);
          oidStr = await this.offeringService.getOidFromHash(tokIndx);
        }
        if (outputTrade) {
          for (let i = 1; i <= numSells; i++) {
            const retValues = await this.getIthSell(
              i,
              tokenPair[0],
              BigNumber.from(tokenPair[1]),
              tokenPair[2],
              BigNumber.from(tokenPair[3]),
            );
            let bourseTx = 'boursetx-';
            bourseTx = bourseTx.concat(
              i.toString(),
              '-',
              tokenPair[0],
              '-',
              assetidStr,
              '-',
              oidStr,
              '-',
              coinSymbol,
            );
            const tradeRecord = [retValues[0], bourseTx, retValues[2]]; // [price, bourse tx, time ]
            allRetValues.push(tradeRecord);
          }
        }
      }
      return allRetValues;
    } catch (error) {
      throw new InternalServerErrorException(
        `Failed in getAssetOrOIdTradeHistory call. ${error}`,
      );
    }
  }

  async getLatestBlockNo(): Promise<number> {
    const latestBlockNo = await this.commonUtils.provider.getBlockNumber();
    return latestBlockNo;
  }

  // Function to perform binary search for a block with a specific timestamp
  async binarySearch(
    targetTime: number,
    low: number,
    high: number,
  ): Promise<any> {
    while (low <= high) {
      const mid = Math.floor((low + high) / 2);
      const midBlock = await this.commonUtils.provider.getBlock(mid);

      if (midBlock.timestamp == targetTime) {
        return mid;
      } else if (midBlock.timestamp < targetTime) {
        low = mid + 1;
      } else {
        high = mid - 1;
      }
    }
    return low;
  }

  // Function to fetch DirectSell events in a block range
  async fetchDirectSellEvents(
    startBlock: number,
    endBlock: number,
  ): Promise<Array<SellData>> {
    // Query the events
    const events = await this.contracts.bourseContract.queryFilter(
      this.contracts.bourseContract.filters.eventDirectSell(),
      startBlock,
      endBlock,
    );

    // Extract parameters from the events
    return events.map((event) => ({
      ta1: event.args?.ta1,
      ti1: event.args?.ti1, // BigNumber
      ta2: event.args?.ta2,
      ti2: event.args?.ti2, // BigNumber
      saleTxNo: event.args?.saleTxNo, // BigNumber
    }));
  }

  async sortSellRecords(
    sells: Array<SellData>,
  ): Promise<Array<[string, string, number, string, number, number]>> {
    const results: Array<[string, string, number, string, number, number]> = [];

    for (const { ta1, ti1, ta2, ti2, saleTxNo } of sells) {
      // Call the `getIthSell` function
      const [price, amount, unixtime] = await this.getIthSell(
        saleTxNo.toNumber(),
        ta1,
        ti1,
        ta2,
        ti2,
      );

      // Combine the results into the desired format
      const assetidHash =
        await this.offeringService.getAssetidHashOfOidHash(ti1);
      const assetid =
        await this.offeringService.getAssetidFromHash(assetidHash);
      const oid = await this.offeringService.getOidFromHash(ti1);
      const priceFormatted = price / Math.pow(10, 18); // assume 18 decimal places in currency token
      const coinSymbol = await this.governanceService.getCoinSymbol(
        ta2,
        BigNumber.from(ti2),
      );
      results.push([
        assetid,
        oid,
        priceFormatted,
        coinSymbol,
        amount,
        unixtime,
      ]);
    }

    // Sort the results in descending order of `unixtime`
    results.sort((a, b) => b[5] - a[5]); // Compare the 8th element (unixtime)

    return results;
  }

  async getSellRecordsInBlockRange(
    startBlock: number,
    endBlock: number,
  ): Promise<Array<[string, string, number, string, number, number]>> {
    const sellEvents = await this.fetchDirectSellEvents(startBlock, endBlock);
    const sortedSellEvents = await this.sortSellRecords(sellEvents);
    return sortedSellEvents;
  }

  async getValidBlockRange(
    intervalLength: number,
    intervalEnd: number,
  ): Promise<{ begBlockNo: number; endBlockNo: number }> {
    // Get the latest block number from the provider
    const latestBlockNo = await this.getLatestBlockNo();

    // Adjust intervalEnd to ensure it doesn't exceed the latest block
    if (latestBlockNo < intervalEnd) {
      intervalEnd = latestBlockNo;
    }

    // Calculate the beginning block, ensuring it doesn't go below 0
    const begBlockNo = Math.max(intervalEnd - intervalLength, 0);

    return { begBlockNo: begBlockNo, endBlockNo: intervalEnd };
  }

  async getOfferingSalesStatistics(
    intervalLength: number, // number of blocks
    intervalEnd: number, // end block number
  ): Promise<Array<[string, string, string, string, number, number]>> {
    const b = await this.getValidBlockRange(intervalLength, intervalEnd);
    const sellRecords = await this.getSellRecordsInBlockRange(
      b.begBlockNo,
      b.endBlockNo,
    );
    const sellStatistics = await this.oidSaleStatistics(sellRecords);
    return sellStatistics;
  }

  async getAssetSalesStatistics(
    intervalLength: number, // number of blocks
    intervalEnd: number, // end block number
  ): Promise<
    Array<[string, string, string, string, string, string, number, number]>
  > {
    const b = await this.getValidBlockRange(intervalLength, intervalEnd);
    //console.log("====>BLOCKRANGE",b.begBlockNo,b.endBlockNo) ;
    const sellRecords = await this.getSellRecordsInBlockRange(
      b.begBlockNo,
      b.endBlockNo,
    );
    //console.log("====>SELLRECORDS",sellRecords) ;
    const sellStatistics = await this.assetidSaleStatistics(sellRecords);
    return sellStatistics;
  }

  async oidSaleStatistics(
    data: Array<[string, string, number, string, number, number]>,
  ): Promise<Array<[string, string, string, string, number, number]>> {
    const results: Array<[string, string, string, string, number, number]> = [];

    // Group data by `oid`
    const groupedData = data.reduce(
      (acc, [assetid, oid, priceFormatted, coinSymbol, amount, unixtime]) => {
        if (!acc[oid]) {
          acc[oid] = [];
        }
        acc[oid].push({
          assetid,
          oid,
          priceFormatted,
          coinSymbol,
          amount,
          unixtime,
        });
        return acc;
      },
      {} as Record<
        string,
        Array<{
          assetid: string;
          oid: string;
          priceFormatted: number;
          coinSymbol: string;
          amount: number;
          unixtime: number;
        }>
      >,
    );

    // Process each group
    for (const oid in groupedData) {
      const group = groupedData[oid];

      const assetid = group[0].assetid; // Assume assetid is the same for the same oid
      const coinSymbol = group[0].coinSymbol; // Assume coinSymbol is the same for the same oid
      const prices = group.map((item) => item.priceFormatted);
      const amounts = group.map((item) => item.amount);

      const minPrice = Math.min(...prices).toFixed(2);

      const totalAmount = amounts.reduce((sum, amount) => sum + amount, 0);
      const count = group.length;

      results.push([assetid, oid, minPrice, coinSymbol, totalAmount, count]);
    }
    return results;
  }

  async assetidSaleStatistics(
    data: Array<[string, string, number, string, number, number]>,
  ): Promise<
    Array<[string, string, string, string, string, string, number, number]>
  > {
    const results: Array<
      [string, string, string, string, string, string, number, number]
    > = [];

    // Group data by `assetid`
    const groupedData = data.reduce(
      (acc, [assetid, oid, priceFormatted, coinSymbol, amount, unixtime]) => {
        if (!acc[assetid]) {
          acc[assetid] = [];
        }
        acc[assetid].push({
          assetid,
          oid,
          priceFormatted,
          coinSymbol,
          amount,
          unixtime,
        });
        return acc;
      },
      {} as Record<
        string,
        Array<{
          assetid: string;
          oid: string;
          priceFormatted: number;
          coinSymbol: string;
          amount: number;
          unixtime: number;
        }>
      >,
    );

    // Process each group
    for (const assetid in groupedData) {
      const group = groupedData[assetid];

      const coinSymbol = group[0].coinSymbol; // Assume coinSymbol is the same for the same assetid
      const prices = group.map((item) => item.priceFormatted);
      const amounts = group.map((item) => item.amount);

      const minPrice = Math.min(...prices).toFixed(2);
      const maxPrice = Math.max(...prices).toFixed(2);
      const avgPrice = (
        prices.reduce((sum, price) => sum + price, 0) / prices.length
      ).toFixed(2);

      const stdDevPrice = Math.sqrt(
        prices.reduce(
          (sum, price) => sum + Math.pow(price - parseFloat(avgPrice), 2),
          0,
        ) / prices.length,
      ).toFixed(2);

      const totalAmount = amounts.reduce((sum, amount) => sum + amount, 0);
      const count = group.length;

      results.push([
        assetid,
        minPrice,
        avgPrice,
        maxPrice,
        stdDevPrice,
        coinSymbol,
        totalAmount,
        count,
      ]);
    }

    return results;
  }

  async getTotalPurchases(): Promise<number> {
    let numPairs: number;

    try {
      numPairs = await this.getNumberOfTradePairs();
    } catch (err) {
      this.logger.error('Error getting number of trading pairs:', err);
      throw new InternalServerErrorException(
        'Failed to retrieve number of trading pairs from Bourse contract',
      );
    }

    let totalBuyBids = 0;

    for (let i = 0; i < numPairs; i++) {
      let ta1: string, ti1: BigNumber, ta2: string, ti2: BigNumber;

      try {
        [ta1, ti1, ta2, ti2] = await this.getIthTradePair(i);
      } catch (err) {
        this.logger.error(`Error getting trading pair at index ${i}:`, err);
        throw new InternalServerErrorException(
          `Failed to retrieve trading pair at index ${i}`,
        );
      }

      try {
        const [numBids] = await this.getTotalNumVolBuys(ta1, ti1, ta2, ti2);
        totalBuyBids += numBids.toNumber();
      } catch (err) {
        this.logger.error(
          `Error getting buy bids for trading pair ${i} (${ta1}:${ti1.toString()} -> ${ta2}:${ti2.toString()}):`,
          err,
        );
        throw new InternalServerErrorException(
          `Failed to retrieve buy bids for trading pair ${i} (${ta1}:${ti1.toString()} -> ${ta2}:${ti2.toString()})`,
        );
      }
    }

    return totalBuyBids;
  }
}
