import { getContractABI } from '@contracts';
import { Injectable, Logger } from '@nestjs/common';
import { ethers } from 'ethers';
import { AddressValidatorService } from './address-validator-service';
import * as dotenv from 'dotenv';
import {
  Bourse,
  DataAccessPayAsYouGo,
  DataAccessPayAsYouUse,
  DataAccessSubscription,
  OfferingToken,
  PaymentToken,
  TradingManagementExecutorFacet,
  TradingManagementStorageFacet,
} from '@contract-types';
const envFile = process.env.NODE_ENV ? `.env.${process.env.NODE_ENV}` : '.env';
console.log(`Loading env file: ${envFile}`);
dotenv.config({ path: envFile, override: true });

type ClientNetwork = 'besu' | 'sepolia' | 'hardhat' | 'localhost';

const BESU_RPC = process.env.BESU_RPC;
const CLIENT_NETWORK: ClientNetwork = process.env.HARDHAT_NETWORK
  ? (process.env.HARDHAT_NETWORK as ClientNetwork)
  : ('besu' as ClientNetwork);
const INFURA_API_KEY = process.env.INFURA_API;

export interface Contracts {
  contractDataAccessPAYG: DataAccessPayAsYouGo;
  contractDataAccessPAYU: DataAccessPayAsYouUse;
  contractDataAccessSubscription: DataAccessSubscription;
  contractOfferingToken: OfferingToken;
  contractPaymentToken: PaymentToken;
  contractGovernance: ethers.Contract;
  bourseContract: ethers.Contract;
  contractTradingManagementExecutor: TradingManagementExecutorFacet;
  contractTradingManagementStorage: TradingManagementStorageFacet;
}

@Injectable()
export class CommonUtils {
  public provider: ethers.providers.JsonRpcProvider;
  private logger = new Logger(CommonUtils.name);

  constructor(
    public readonly addressValidatorService: AddressValidatorService,
  ) {
    switch (CLIENT_NETWORK) {
      case 'besu':
        this.provider = new ethers.providers.JsonRpcProvider(
          `http://${BESU_RPC}`,
        );
        this.logger.log(`Selected BESU network`);
        break;
      case 'sepolia':
        this.provider = new ethers.providers.JsonRpcProvider(
          `https://sepolia.infura.io/v3/${INFURA_API_KEY}`,
        );
        this.logger.log(`Selected SEPOLIA network`);
        break;
      case 'hardhat':
      case 'localhost':
        this.provider = new ethers.providers.JsonRpcProvider(
          `http://127.0.0.1:8545`,
        );
        this.logger.log(`Selected HARDHAT network`);
        break;
      default:
        throw new Error('Unsupported networ set in .env HARDHAT_NETWORK');
    }
  }

  decimalToWei(value: number | ethers.BigNumber): ethers.BigNumber {
    if (typeof value === 'undefined') {
      return ethers.BigNumber.from(0);
    }
    return ethers.utils.parseEther(value.toString());
  }

  weiToDecimals(valueInWei: ethers.BigNumber): number {
    return parseFloat(ethers.utils.formatEther(valueInWei));
  }

  isValidEthAddress(addr): boolean {
    if (/^0x[0-9a-fA-F]{40}$/.test(addr)) return true;
    return false;
  }

  isValidPrivateKey(addr): boolean {
    if (/^0x[0-9a-fA-F]{64}$/.test(addr)) return true;
    return false;
  }

  getContracts(): Contracts {
    return {
      contractDataAccessPAYG: new ethers.Contract(
        this.addressValidatorService.contractAddresses.contractAddressDataAccessPAYG,
        getContractABI('DataAccessPayAsYouGo'),
        this.provider,
      ) as DataAccessPayAsYouGo,
      contractDataAccessPAYU: new ethers.Contract(
        this.addressValidatorService.contractAddresses.contractAddressDataAccessPAYU,
        getContractABI('DataAccessPayAsYouUse'),
        this.provider,
      ) as DataAccessPayAsYouUse,
      contractDataAccessSubscription: new ethers.Contract(
        this.addressValidatorService.contractAddresses.contractAddressDataAccessSubscription,
        getContractABI('DataAccessSubscription'),
        this.provider,
      ) as DataAccessSubscription,
      contractOfferingToken: new ethers.Contract(
        this.addressValidatorService.contractAddresses.contractAddressOfferingToken,
        getContractABI('OfferingToken'),
        this.provider,
      ) as OfferingToken,
      contractPaymentToken: new ethers.Contract(
        this.addressValidatorService.contractAddresses.contractAddressPaymentToken,
        getContractABI('PaymentToken'),
        this.provider,
      ) as PaymentToken,
      contractGovernance: new ethers.Contract(
        this.addressValidatorService.contractAddresses.contractAddressGovernance,
        getContractABI('GovernanceFacet'),
        this.provider,
      ),
      bourseContract: new ethers.Contract(
        this.addressValidatorService.contractAddresses.contractAddressBourse,
        getContractABI('BourseFacet'),
        this.provider,
      ) as Bourse,
      contractTradingManagementExecutor: new ethers.Contract(
        this.addressValidatorService.contractAddresses.contractAddressTradingManagement,
        getContractABI('TradingManagementExecutorFacet'),
        this.provider,
      ) as TradingManagementExecutorFacet,
      contractTradingManagementStorage: new ethers.Contract(
        this.addressValidatorService.contractAddresses.contractAddressTradingManagement,
        getContractABI('TradingManagementStorageFacet'),
        this.provider,
      ) as TradingManagementStorageFacet,
    };
  }
}
