import * as fs from 'fs';
import * as path from 'path';
import findPackageJsonDir from './findRootDir';

export interface ContractAddresses {
  contractAddressPaymentToken: string;
  contractAddressOfferingToken: string;
  contractAddressBourse: string;
  contractAddressDataAccessPAYG: string;
  contractAddressDataAccessPAYU: string;
  contractAddressDataAccessSubscription: string;
  contractAddressGovernance: string;
  contractAddressEscrow: string;
  contractAddressTradingManagement: string;
}

export function getContractAddresses(network: string): ContractAddresses {
  const rootDir = findPackageJsonDir(__dirname);
  const jsonDir = path.join(rootDir, 'libs/contracts/contract-addresses.json');
  try {
    const fileContent = fs.readFileSync(jsonDir, 'utf8');
    const addressesMap = JSON.parse(fileContent);
    if (!addressesMap[network]) {
      throw new Error(`No addresses found for network: ${network}`);
    }
    return addressesMap[network];
  } catch (error) {
    console.error(`Failed to get contract addresses: ${error.message}`);
    return null;
  }
}

export function saveContractAddresses(
  network: string,
  addresses: Record<string, string>,
): void {
  const rootDir = findPackageJsonDir(__dirname);
  const jsonDir = path.join(rootDir, 'libs/contracts/contract-addresses.json');
  const addressesMap = JSON.parse(fs.readFileSync(jsonDir, 'utf8'));
  addressesMap[network] = { ...addressesMap[network], ...addresses };
  fs.writeFileSync(jsonDir, JSON.stringify(addressesMap, null, 2));
  console.log(
    `------------------------------------------------
    Saved contract addresses for ${network} network
    `,
    JSON.stringify(addressesMap[network], null, 2),
  );
  console.log('------------------------------------------------');
}

export interface FacetInfo {
  publishData: string;
  facetName: string;
  facetAddress: string;
}

export type DiamondInfo = {
  name: string;
  diamondAddress: string;
  diamondInitAddress: string;
  ownership: string;
  publishData: string;
};

export type DiamondFullInfo = DiamondInfo & {
  facets: Record<string, FacetInfo[]>;
};

type DiamondFullInfoMap = Record<string, DiamondFullInfo[]>;

type DiamondFullInfoMapPerNetwork = {
  [network: string]: DiamondFullInfoMap;
};

export function saveDiamondInfo(
  network: string,
  toSaveDiamondFullInfo: DiamondFullInfo,
): DiamondFullInfoMapPerNetwork {
  const rootDir = findPackageJsonDir(__dirname);
  const jsonDir = path.join(rootDir, 'libs/contracts/diamonds-info.json');

  // Check if the file exists, and if not, create it with an empty object
  if (!fs.existsSync(jsonDir)) {
    fs.writeFileSync(jsonDir, JSON.stringify({}));
  }

  const diamondInfoResult: DiamondFullInfoMapPerNetwork = JSON.parse(
    fs.readFileSync(jsonDir, 'utf8'),
  );

  if (!diamondInfoResult[network]) {
    diamondInfoResult[network] = {};
  }

  // Initialize the diamond array if it doesn't exist
  if (!diamondInfoResult[network][toSaveDiamondFullInfo.name]) {
    diamondInfoResult[network][toSaveDiamondFullInfo.name] = [];
  }

  // Find the index of the existing diamond (if any)
  const foundDiamondIndex = diamondInfoResult[network][
    toSaveDiamondFullInfo.name
  ].findIndex(
    (el) => el.diamondAddress === toSaveDiamondFullInfo.diamondAddress,
  );

  if (foundDiamondIndex === -1) {
    // **New Diamond Case:**
    // Add the new diamond info to the array
    diamondInfoResult[network][toSaveDiamondFullInfo.name].push({
      name: toSaveDiamondFullInfo.name,
      diamondAddress: toSaveDiamondFullInfo.diamondAddress,
      diamondInitAddress: toSaveDiamondFullInfo.diamondInitAddress,
      ownership: toSaveDiamondFullInfo.ownership,
      publishData:
        toSaveDiamondFullInfo.publishData ?? new Date().toISOString(),
      facets: { ...toSaveDiamondFullInfo.facets },
    });
  } else {
    // **Upgrade Case:**
    // Get the existing diamond info
    const existingDiamond =
      diamondInfoResult[network][toSaveDiamondFullInfo.name][foundDiamondIndex];

    // Get existing facets
    const existingFacets = existingDiamond.facets;

    // Initialize updatedFacets as a copy of existingFacets
    const updatedFacets: Record<string, FacetInfo[]> = { ...existingFacets };

    // **Merge the new facet info with the existing facets**
    for (const [facetName, facetInfos] of Object.entries(
      toSaveDiamondFullInfo.facets,
    )) {
      if (updatedFacets[facetName]) {
        // **Append new facet info to existing array only if addresses are different **
        for (const facetInfo of facetInfos) {
          const foundFacetIndex = updatedFacets[facetName].findIndex(
            (el) => el.facetAddress === facetInfo.facetAddress,
          );
          if (foundFacetIndex === -1) {
            updatedFacets[facetName].push(facetInfo);
          }
        }
      } else {
        // **Add new facet with its facet info array**
        updatedFacets[facetName] = facetInfos;
      }
    }

    // **Update the existing diamond info with updated facets**
    diamondInfoResult[network][toSaveDiamondFullInfo.name][foundDiamondIndex] =
      {
        ...existingDiamond,
        facets: updatedFacets,
        publishData:
          toSaveDiamondFullInfo.publishData ?? new Date().toISOString(),
      };
  }

  // **Save the updated diamond info to the JSON file**
  fs.writeFileSync(jsonDir, JSON.stringify(diamondInfoResult, null, 2));
  console.log(
    `------------------------------------------------
    Saved diamond info for ${toSaveDiamondFullInfo.name} on ${network} network
    `,
    // JSON.stringify(
    //   diamondInfoResult[network][toSaveDiamondFullInfo.name],
    //   null,
    //   2,
    // ),
  );
  console.log('------------------------------------------------');
  return diamondInfoResult;
}

export function getDeployedDATContractAddress(
  network: string,
  tokenType: string,
): string {
  const addresses = getContractAddresses(network);
  if (!addresses) {
    throw new Error(`No addresses found for network: ${network}`);
  }

  switch (tokenType.toUpperCase()) {
    case 'SUB':
      return addresses.contractAddressDataAccessSubscription;
    case 'PAYG':
      return addresses.contractAddressDataAccessPAYG;
    case 'PAYU':
      return addresses.contractAddressDataAccessPAYU;
    default:
      throw new Error(`Invalid token type provided: ${tokenType}`);
  }
}

export function readLastDeployedDiamondInfo(
  network: string,
  diamondName: string,
): DiamondFullInfo {
  const rootDir = findPackageJsonDir(__dirname);
  const jsonDir = path.join(rootDir, 'libs/contracts/diamonds-info.json');

  // Check if the file exists, and if not, create it with an empty object
  if (!fs.existsSync(jsonDir)) {
    fs.writeFileSync(jsonDir, JSON.stringify({}));
  }

  const diamondInfoResult: DiamondFullInfoMapPerNetwork = JSON.parse(
    fs.readFileSync(jsonDir, 'utf8'),
  );

  if (!diamondInfoResult[network]) {
    diamondInfoResult[network] = {};
  }
  if (!diamondInfoResult[network][diamondName]) {
    throw new Error(`No saved infofor diamond found with name: ${diamondName}`);
  }
  const getLast = diamondInfoResult[network][diamondName].length - 1;
  return diamondInfoResult[network][diamondName][getLast];
}
