// SPDX-License-Identifier: MIT
pragma solidity ^0.8.17;

import '@openzeppelin/contracts-upgradeable/token/ERC1155/ERC1155Upgradeable.sol';
import '@openzeppelin/contracts-upgradeable/proxy/utils/Initializable.sol';
import '@openzeppelin/contracts-upgradeable/access/OwnableUpgradeable.sol';
import '@openzeppelin/contracts-upgradeable/security/PausableUpgradeable.sol';
import './Bourse.sol';
import './PaymentToken.sol';
import './OfferingToken.sol';

abstract contract BaseDataAccess is
    Initializable,
    ERC1155Upgradeable,
    OwnableUpgradeable,
    PausableUpgradeable
{
    // amount of instances of a data access token id
    mapping(uint256 => uint256) public dataAccessTokenAmount;
    string public name;
    string public symbol;

    event TokenDataAccessMinted(
        address indexed owner,
        uint256 tokenId,
        bytes data
    );

    event DataAccessBurned(address indexed owner, uint256 tokenId, bytes data);

    function initialize(
        string memory _name,
        string memory _symbol,
        string memory _uri
    ) public virtual initializer {
        __ERC1155_init(_uri);
        __Ownable_init();
        __Pausable_init();
        name = _name;
        symbol = _symbol;
    }

    function setURI(string memory newURI) public onlyOwner {
        _setURI(newURI);
    }

    function mintAndApprove(
        uint256 _assetId,
        address _operator,
        bytes memory _data
    ) external virtual whenNotPaused {
        // Todo: Only should be called by TradingManagementExecutorFacet

        //minting and directselling to the consumer/data buyer

        address minter = msg.sender;
        _mint(minter, _assetId, 1, '');
        emit TokenDataAccessMinted(minter, _assetId, _data);
        _setApprovalForAll(minter, address(_operator), true);
        //increasing the amount on this id
        ++dataAccessTokenAmount[_assetId];
    }

    // Todo: Only should be called by TradingManagementExecutorFacet

    function burn(
        address _from,
        uint256 _assetId,
        bytes memory _data
    ) internal virtual {
        _burn(_from, _assetId, 1);
        --dataAccessTokenAmount[_assetId];
        emit DataAccessBurned(_from, _assetId, _data);
    }

    function getDataAccessTokenAmount(
        uint256 _assetId
    ) public view returns (uint256) {
        return dataAccessTokenAmount[_assetId];
    }

    function pause() external onlyOwner {
        _pause();
    }

    function unpause() external onlyOwner {
        _unpause();
    }
}
