pragma solidity ^0.8.17;

// SPDX-License-Identifier: AGPL-3.0-or-later

// Note: Because of contract code size limit ( https://eips.ethereum.org/EIPS/eip-170 )
//       code has been divided into components in the form of libraries

import '@openzeppelin/contracts/token/ERC20/IERC20.sol';
import '@openzeppelin/contracts/token/ERC1155/IERC1155.sol';
import '@openzeppelin/contracts/token/ERC721/IERC721.sol';
import '@openzeppelin/contracts/utils/introspection/ERC165.sol';

import './diamond/Diamond.sol';

interface ESCROW {
    function deposit(address buyer, uint256 amnt, uint256 indx) external;
}

library AVLTreeLib {
    //////////////////////////// AVL Tree Library //////////////////////////

    struct Node {
        // Transaction node
        uint price;
        uint amount;
        address owner;
        uint8 status;
        uint left; // tree left pointer (used in bidding)
        uint right; // tree right pointer (used in bidding)
        uint height; // tree height
        uint splitprev;
        uint txtime; // transaction time
    }

    struct AVLTree {
        Node[] nodes;
        uint treeroot;
        mapping(address => uint[]) ownerbidlist;
    }

    function AVLTreeLibInit(AVLTree storage self) public {
        self.treeroot = 0;
        self.nodes.push(
            Node({
                price: 0,
                amount: 0,
                owner: address(0),
                left: 0,
                right: 0,
                height: 0,
                splitprev: 0,
                txtime: 0,
                status: 0
            })
        );
    }

    function newNode(
        AVLTree storage self,
        uint _price,
        uint _amnt,
        address owner,
        uint prev
    ) public returns (uint) {
        uint btime = block.timestamp;
        self.nodes.push(
            Node({
                price: _price,
                amount: _amnt,
                owner: owner,
                left: 0,
                right: 0,
                height: 1,
                splitprev: prev,
                txtime: btime,
                status: 0
            })
        );
        self.ownerbidlist[owner].push(self.nodes.length - 1);
        return (self.nodes.length - 1);
    }

    function height(
        AVLTree storage self,
        uint node
    ) public view returns (uint) {
        return ((node == 0) ? 0 : self.nodes[node].height);
    }

    function getMax(uint a, uint b) public pure returns (uint) {
        return ((a > b) ? a : b);
    }

    function leftRotate(AVLTree storage self, uint node) public returns (uint) {
        uint noderight = self.nodes[node].right;
        uint rightleft = self.nodes[noderight].left;

        self.nodes[noderight].left = node;
        self.nodes[node].right = rightleft;

        self.nodes[node].height =
            getMax(
                height(self, self.nodes[node].left),
                height(self, self.nodes[node].right)
            ) +
            1;
        self.nodes[noderight].height =
            getMax(
                height(self, self.nodes[noderight].left),
                height(self, self.nodes[noderight].right)
            ) +
            1;

        return (noderight);
    }

    function rightRotate(
        AVLTree storage self,
        uint node
    ) public returns (uint) {
        uint nodeleft = self.nodes[node].left;
        uint leftright = self.nodes[nodeleft].right;

        self.nodes[nodeleft].right = node;
        self.nodes[node].left = leftright;

        self.nodes[node].height =
            getMax(
                height(self, self.nodes[node].left),
                height(self, self.nodes[node].right)
            ) +
            1;
        self.nodes[nodeleft].height =
            getMax(
                height(self, self.nodes[nodeleft].left),
                height(self, self.nodes[nodeleft].right)
            ) +
            1;

        return (nodeleft);
    }

    function treeBalanceFactor(
        AVLTree storage self,
        uint node
    ) public view returns (int) {
        return (
            (node == 0)
                ? int(0)
                : (int(height(self, self.nodes[node].left)) -
                    int(height(self, self.nodes[node].right)))
        );
    }

    function insertNode(
        AVLTree storage self,
        uint node,
        uint price,
        uint amnt,
        address owner,
        uint prev
    ) public returns (uint) {
        if (node == 0) return (newNode(self, price, amnt, owner, prev));

        if (price < self.nodes[node].price)
            self.nodes[node].left = insertNode(
                self,
                self.nodes[node].left,
                price,
                amnt,
                owner,
                prev
            );
        else if (price > self.nodes[node].price)
            self.nodes[node].right = insertNode(
                self,
                self.nodes[node].right,
                price,
                amnt,
                owner,
                prev
            );
        else return node;

        self.nodes[node].height =
            1 +
            getMax(
                height(self, self.nodes[node].left),
                height(self, self.nodes[node].right)
            );

        int balfactor = treeBalanceFactor(self, node);
        if (balfactor > 1 && price < self.nodes[self.nodes[node].left].price)
            return rightRotate(self, node);

        if (balfactor < -1 && price > self.nodes[self.nodes[node].right].price)
            return leftRotate(self, node);

        if (balfactor > 1 && price > self.nodes[self.nodes[node].left].price) {
            self.nodes[node].left = leftRotate(self, self.nodes[node].left);
            return rightRotate(self, node);
        }

        if (
            balfactor < -1 && price < self.nodes[self.nodes[node].right].price
        ) {
            self.nodes[node].right = rightRotate(self, self.nodes[node].right);
            return leftRotate(self, node);
        }

        return (node);
    }

    function minPriceNode(
        AVLTree storage self,
        uint node
    ) public view returns (uint) {
        uint current = node;

        while (self.nodes[current].left != 0)
            current = self.nodes[current].left;

        return current;
    }

    function minPrice(AVLTree storage self) public view returns (uint) {
        uint minnode = minPriceNode(self, self.treeroot);
        return (self.nodes[minnode].price);
    }

    function maxPrice(AVLTree storage self) public view returns (uint) {
        uint maxnode = maxPriceNode(self, self.treeroot);
        return (self.nodes[maxnode].price);
    }

    function maxPriceNode(
        AVLTree storage self,
        uint node
    ) public view returns (uint) {
        uint current = node;

        while (self.nodes[current].right != 0)
            current = self.nodes[current].right;

        return current;
    }

    function deleteNode(
        AVLTree storage self,
        uint root,
        uint price
    ) public returns (uint) {
        if (root == 0) return root;

        if (price < self.nodes[root].price)
            self.nodes[root].left = deleteNode(
                self,
                self.nodes[root].left,
                price
            );
        else if (price > self.nodes[root].price)
            self.nodes[root].right = deleteNode(
                self,
                self.nodes[root].right,
                price
            );
        else {
            if ((self.nodes[root].left == 0) || (self.nodes[root].right == 0)) {
                // uint temp = self.nodes[root].left ? self.nodes[root].left : self.nodes[root].right;
                uint temp = (self.nodes[root].left == 0)
                    ? self.nodes[root].right
                    : self.nodes[root].left;
                if (temp == 0) {
                    temp = root;
                    root = 0;
                } else self.nodes[root] = self.nodes[temp];
            } else {
                uint temp = minPriceNode(self, self.nodes[root].right);
                self.nodes[root].price = self.nodes[temp].price;
                self.nodes[root].right = deleteNode(
                    self,
                    self.nodes[root].right,
                    self.nodes[temp].price
                );
            }
        }

        if (root == 0) return root;

        self.nodes[root].height =
            1 +
            getMax(
                height(self, self.nodes[root].left),
                height(self, self.nodes[root].right)
            );

        int balfactor = treeBalanceFactor(self, root);
        if (
            balfactor > 1 && treeBalanceFactor(self, self.nodes[root].left) >= 0
        ) return rightRotate(self, root);

        if (
            balfactor > 1 && treeBalanceFactor(self, self.nodes[root].left) < 0
        ) {
            self.nodes[root].left = leftRotate(self, self.nodes[root].left);
            return rightRotate(self, root);
        }

        if (
            balfactor < -1 &&
            treeBalanceFactor(self, self.nodes[root].right) <= 0
        ) return leftRotate(self, root);

        if (
            balfactor < -1 &&
            treeBalanceFactor(self, self.nodes[root].right) > 0
        ) {
            self.nodes[root].right = rightRotate(self, self.nodes[root].right);
            return leftRotate(self, root);
        }
        return root;
    }

    function treeInsertNode(
        AVLTree storage self,
        uint price,
        uint amnt,
        address owner,
        uint prev
    ) public {
        self.treeroot = insertNode(
            self,
            self.treeroot,
            price,
            amnt,
            owner,
            prev
        );
    }

    function treeDeleteNode(AVLTree storage self, uint price) public {
        self.treeroot = deleteNode(self, self.treeroot, price);
    }
}

library TokenUtilsLib {
    enum TypeToken {
        undefinedToken,
        typeERC20,
        typeERC1155,
        typeERC721
    }

    //////////////////////// interface IDs.  (ERC165 -  https://eips.ethereum.org/EIPS/eip-165 )
    bytes4 constant interfaceIdERC1155 = 0xd9b67a26;
    bytes4 constant interfaceIdERC721 = 0x80ac58cd;
    // ERC20 is not required to implement ERC165

    // Please see :
    // https://github.com/OpenZeppelin/openzeppelin-contracts/blob/master/contracts/utils/introspection/ERC165Checker.sol
    // https://forum.openzeppelin.com/t/erc165checker-is-it-required-to-check-support-for-erc-165-and-any-given-interface-or-just-the-latter/29821/3
    function supportsInterface2(
        address accaddr,
        bytes4 interfaceID
    ) public view returns (bool) {
        bytes memory data = abi.encodeWithSelector(
            ERC165.supportsInterface.selector,
            interfaceID
        );
        (bool success, bytes memory result) = accaddr.staticcall{gas: 30000}(
            data
        );
        return ((result.length >= 0x20) &&
            success &&
            abi.decode(result, (bool)));
    }

    function getTokenType(address ta) public view returns (TypeToken) {
        if (supportsInterface2(ta, interfaceIdERC1155)) {
            return (TypeToken.typeERC1155);
        } else if (supportsInterface2(ta, interfaceIdERC721)) {
            return (TypeToken.typeERC721);
        }
        // otherwise assume it is ERC20
        return (TypeToken.typeERC20);
    }

    function checkAllowance(
        TypeToken toktype,
        address ta,
        uint ti,
        address useraddr,
        uint amnt
    ) public view returns (bool) {
        if (toktype == TypeToken.typeERC20) {
            if (IERC20(ta).allowance(useraddr, address(this)) >= amnt)
                return (true);
        } else if (toktype == TypeToken.typeERC1155) {
            if (IERC1155(ta).isApprovedForAll(useraddr, address(this))) {
                // check balance
                if (IERC1155(ta).balanceOf(useraddr, ti) >= amnt) return (true);
            }
        } else if (toktype == TypeToken.typeERC721) {
            if (IERC721(ta).getApproved(ti) == address(this)) return (true);
        }

        return (false);
    }

    function tokenTransfers(
        LibBourseStorage.BourseData storage self,
        address seller,
        address buyer,
        LibBourseStorage.TokenPair memory p,
        uint amnt,
        uint price
    ) public {
        if (self.typeoftoken[p.ta1] == TypeToken.typeERC20) {
            require(IERC20(p.ta1).transferFrom(msg.sender, buyer, amnt));
        } else if (self.typeoftoken[p.ta1] == TypeToken.typeERC1155) {
            IERC1155(p.ta1).safeTransferFrom(
                msg.sender,
                buyer,
                p.ti1,
                amnt,
                ''
            );
        } else if (self.typeoftoken[p.ta1] == TypeToken.typeERC721) {
            IERC721(p.ta1).transferFrom(msg.sender, buyer, p.ti1);
        } else {
            revert();
        }

        if (self.typeoftoken[p.ta2] == TypeToken.typeERC20) {
            require(IERC20(p.ta2).transferFrom(buyer, seller, amnt * price));
        } else if (self.typeoftoken[p.ta2] == TypeToken.typeERC1155) {
            IERC1155(p.ta2).safeTransferFrom(
                buyer,
                seller,
                p.ti2,
                amnt * price,
                ''
            );
        } else if (self.typeoftoken[p.ta2] == TypeToken.typeERC721) {
            IERC721(p.ta2).transferFrom(buyer, seller, p.ti2);
        } else {
            revert();
        }
    }
}

library LibBourseStorage {
    bytes32 constant BOURSE_STORAGE_POSITION =
        keccak256('diamond.storage.BourseStorage.v1');

    enum PayTypeStatus {
        SUBSCRIPTION,
        PAY_AS_YOU_GO,
        PAY_AS_YOU_USE,
        DIRECTSALE,
        BID_ACTIVE,
        BID_PARTIAL_MATCHED,
        BID_MATCHED,
        BID_CANCELLED
    }

    struct TokenPair {
        address ta1;
        uint ti1;
        address ta2;
        uint ti2;
    }

    struct TradePair {
        AVLTreeLib.AVLTree buybids;
        AVLTreeLib.AVLTree sellbids;
        uint numbuybids;
        uint numsellbids;
        uint volbuybids;
        uint volsellbids;
        uint ascendcounter;
        uint descendcounter;
        TokenPair tp;
    }

    struct BourseData {
        mapping(address => mapping(uint => mapping(address => mapping(uint => TradePair)))) tradepairs;
        TokenPair[] tokenpairlist;
        address escrowaddr;
        mapping(address => mapping(uint256 => bool)) currencycoin;
        mapping(address => uint) allowanceblocked;
        uint blockedduration; // in seconds
        mapping(address => TokenUtilsLib.TypeToken) typeoftoken;
        uint COUNTERBITS; // used to implement first come first served in bid processing
    }

    function bourseStorage() internal pure returns (BourseData storage bd) {
        bytes32 position = BOURSE_STORAGE_POSITION;
        // solhint-disable-next-line no-inline-assembly
        assembly {
            bd.slot := position
        }
    }
}

library BourseLib {
    using AVLTreeLib for AVLTreeLib.AVLTree;

    ////////// Events
    event eventSingleMatch(
        uint indexed buyerbidno,
        uint indexed sellerbidno,
        uint amnt
    );

    ////////// Functions

    function allowedCurrencyCoin(
        LibBourseStorage.BourseData storage self,
        address ta,
        uint ti
    ) public view returns (bool) {
        return (self.currencycoin[ta][ti]);
    }

    function existsTradePair(
        LibBourseStorage.BourseData storage self,
        address ta1,
        uint ti1,
        address ta2,
        uint ti2
    ) public view returns (bool) {
        return (
            (self.tradepairs[ta1][ti1][ta2][ti2].ascendcounter == 0)
                ? false
                : true
        );
    }

    function singleMatch(
        LibBourseStorage.BourseData storage self,
        LibBourseStorage.TokenPair memory p
    ) public returns (bool) {
        uint bn;
        uint sn;
        uint amnt;
        int remain;

        require(allowedCurrencyCoin(self, p.ta2, p.ti2));
        require(existsTradePair(self, p.ta1, p.ti1, p.ta2, p.ti2));

        bn = self.tradepairs[p.ta1][p.ti1][p.ta2][p.ti2].buybids.maxPriceNode(
            self.tradepairs[p.ta1][p.ti1][p.ta2][p.ti2].buybids.treeroot
        );
        sn = self.tradepairs[p.ta1][p.ti1][p.ta2][p.ti2].sellbids.minPriceNode(
            self.tradepairs[p.ta1][p.ti1][p.ta2][p.ti2].sellbids.treeroot
        );

        require(
            (self
            .tradepairs[p.ta1][p.ti1][p.ta2][p.ti2].buybids.nodes[bn].price >>
                self.COUNTERBITS) >=
                (self
                .tradepairs[p.ta1][p.ti1][p.ta2][p.ti2]
                    .sellbids
                    .nodes[sn]
                    .price >> self.COUNTERBITS)
        );

        if (
            !TokenUtilsLib.checkAllowance(
                self.typeoftoken[p.ta2],
                p.ta2,
                p.ti2,
                self
                .tradepairs[p.ta1][p.ti1][p.ta2][p.ti2].buybids.nodes[bn].owner,
                self
                .tradepairs[p.ta1][p.ti1][p.ta2][p.ti2]
                    .buybids
                    .nodes[bn]
                    .amount *
                    (self
                    .tradepairs[p.ta1][p.ti1][p.ta2][p.ti2]
                        .buybids
                        .nodes[bn]
                        .price >> self.COUNTERBITS)
            )
        ) {
            // if no allowance, delete bid and block the owner
            self.tradepairs[p.ta1][p.ti1][p.ta2][p.ti2].buybids.treeDeleteNode(
                self
                .tradepairs[p.ta1][p.ti1][p.ta2][p.ti2].buybids.nodes[bn].price
            );
            self.allowanceblocked[
                self
                .tradepairs[p.ta1][p.ti1][p.ta2][p.ti2].buybids.nodes[bn].owner
            ] = block.timestamp;
            return (false);
        }

        if (
            !TokenUtilsLib.checkAllowance(
                self.typeoftoken[p.ta1],
                p.ta1,
                p.ti1,
                self
                .tradepairs[p.ta1][p.ti1][p.ta2][p.ti2]
                    .sellbids
                    .nodes[sn]
                    .owner,
                self
                .tradepairs[p.ta1][p.ti1][p.ta2][p.ti2]
                    .sellbids
                    .nodes[sn]
                    .amount
            )
        ) {
            // if no allowance, delete bid and block owner
            self.tradepairs[p.ta1][p.ti1][p.ta2][p.ti2].sellbids.treeDeleteNode(
                self
                .tradepairs[p.ta1][p.ti1][p.ta2][p.ti2].sellbids.nodes[sn].price
            );
            self.allowanceblocked[
                self
                .tradepairs[p.ta1][p.ti1][p.ta2][p.ti2].sellbids.nodes[sn].owner
            ] = block.timestamp;
            return (false);
        }

        remain =
            int(
                self
                .tradepairs[p.ta1][p.ti1][p.ta2][p.ti2].buybids.nodes[bn].amount
            ) -
            int(
                self
                .tradepairs[p.ta1][p.ti1][p.ta2][p.ti2]
                    .sellbids
                    .nodes[sn]
                    .amount
            );
        self.tradepairs[p.ta1][p.ti1][p.ta2][p.ti2].buybids.treeDeleteNode(
            self.tradepairs[p.ta1][p.ti1][p.ta2][p.ti2].buybids.nodes[bn].price
        );
        self.tradepairs[p.ta1][p.ti1][p.ta2][p.ti2].sellbids.treeDeleteNode(
            self.tradepairs[p.ta1][p.ti1][p.ta2][p.ti2].sellbids.nodes[sn].price
        );

        if (remain > 0) {
            self
            .tradepairs[p.ta1][p.ti1][p.ta2][p.ti2]
                .buybids
                .nodes[bn]
                .status = uint8(
                LibBourseStorage.PayTypeStatus.BID_PARTIAL_MATCHED
            );
            self
            .tradepairs[p.ta1][p.ti1][p.ta2][p.ti2]
                .sellbids
                .nodes[sn]
                .status = uint8(LibBourseStorage.PayTypeStatus.BID_MATCHED);
            amnt = self
            .tradepairs[p.ta1][p.ti1][p.ta2][p.ti2].sellbids.nodes[sn].amount;
            self.tradepairs[p.ta1][p.ti1][p.ta2][p.ti2].buybids.treeInsertNode(
                self
                .tradepairs[p.ta1][p.ti1][p.ta2][p.ti2].buybids.nodes[bn].price,
                uint(remain),
                self
                .tradepairs[p.ta1][p.ti1][p.ta2][p.ti2].buybids.nodes[bn].owner,
                bn
            );
        } else if (remain < 0) {
            self
            .tradepairs[p.ta1][p.ti1][p.ta2][p.ti2]
                .buybids
                .nodes[bn]
                .status = uint8(LibBourseStorage.PayTypeStatus.BID_MATCHED);
            self
            .tradepairs[p.ta1][p.ti1][p.ta2][p.ti2]
                .sellbids
                .nodes[sn]
                .status = uint8(
                LibBourseStorage.PayTypeStatus.BID_PARTIAL_MATCHED
            );
            amnt = self
            .tradepairs[p.ta1][p.ti1][p.ta2][p.ti2].buybids.nodes[bn].amount;
            self.tradepairs[p.ta1][p.ti1][p.ta2][p.ti2].sellbids.treeInsertNode(
                self
                .tradepairs[p.ta1][p.ti1][p.ta2][p.ti2]
                    .sellbids
                    .nodes[sn]
                    .price,
                uint(-remain),
                self
                .tradepairs[p.ta1][p.ti1][p.ta2][p.ti2]
                    .sellbids
                    .nodes[sn]
                    .owner,
                sn
            );
        } else {
            self
            .tradepairs[p.ta1][p.ti1][p.ta2][p.ti2]
                .buybids
                .nodes[bn]
                .status = uint8(LibBourseStorage.PayTypeStatus.BID_MATCHED);
            self
            .tradepairs[p.ta1][p.ti1][p.ta2][p.ti2]
                .sellbids
                .nodes[sn]
                .status = uint8(LibBourseStorage.PayTypeStatus.BID_MATCHED);
            amnt = self
            .tradepairs[p.ta1][p.ti1][p.ta2][p.ti2].buybids.nodes[bn].amount;
        }

        if (self.typeoftoken[p.ta2] == TokenUtilsLib.TypeToken.typeERC20) {
            require(
                IERC20(p.ta2).transferFrom(
                    self
                    .tradepairs[p.ta1][p.ti1][p.ta2][p.ti2]
                        .sellbids
                        .nodes[sn]
                        .owner,
                    self
                    .tradepairs[p.ta1][p.ti1][p.ta2][p.ti2]
                        .buybids
                        .nodes[bn]
                        .owner,
                    amnt
                )
            );
        } else if (
            self.typeoftoken[p.ta2] == TokenUtilsLib.TypeToken.typeERC1155
        ) {
            IERC1155(p.ta2).safeTransferFrom(
                self
                .tradepairs[p.ta1][p.ti1][p.ta2][p.ti2]
                    .sellbids
                    .nodes[sn]
                    .owner,
                self
                .tradepairs[p.ta1][p.ti1][p.ta2][p.ti2].buybids.nodes[bn].owner,
                p.ti2,
                amnt,
                ''
            );
        } else if (
            self.typeoftoken[p.ta2] == TokenUtilsLib.TypeToken.typeERC721
        ) {
            IERC721(p.ta2).transferFrom(
                self
                .tradepairs[p.ta1][p.ti1][p.ta2][p.ti2]
                    .sellbids
                    .nodes[sn]
                    .owner,
                self
                .tradepairs[p.ta1][p.ti1][p.ta2][p.ti2].buybids.nodes[bn].owner,
                p.ti2
            );
        } else {
            revert();
        }

        if (self.typeoftoken[p.ta1] == TokenUtilsLib.TypeToken.typeERC20) {
            require(
                IERC20(p.ta1).transferFrom(
                    self
                    .tradepairs[p.ta1][p.ti1][p.ta2][p.ti2]
                        .buybids
                        .nodes[bn]
                        .owner,
                    self
                    .tradepairs[p.ta1][p.ti1][p.ta2][p.ti2]
                        .sellbids
                        .nodes[sn]
                        .owner,
                    (amnt *
                        ((self
                        .tradepairs[p.ta1][p.ti1][p.ta2][p.ti2]
                            .buybids
                            .nodes[bn]
                            .price >> self.COUNTERBITS) +
                            (self
                            .tradepairs[p.ta1][p.ti1][p.ta2][p.ti2]
                                .sellbids
                                .nodes[sn]
                                .price >> self.COUNTERBITS))) / 2
                )
            );
        } else if (
            self.typeoftoken[p.ta2] == TokenUtilsLib.TypeToken.typeERC1155
        ) {
            IERC1155(p.ta1).safeTransferFrom(
                self
                .tradepairs[p.ta1][p.ti1][p.ta2][p.ti2].buybids.nodes[bn].owner,
                self
                .tradepairs[p.ta1][p.ti1][p.ta2][p.ti2]
                    .sellbids
                    .nodes[sn]
                    .owner,
                p.ti1,
                (amnt *
                    ((self
                    .tradepairs[p.ta1][p.ti1][p.ta2][p.ti2]
                        .buybids
                        .nodes[bn]
                        .price >> self.COUNTERBITS) +
                        (self
                        .tradepairs[p.ta1][p.ti1][p.ta2][p.ti2]
                            .sellbids
                            .nodes[sn]
                            .price >> self.COUNTERBITS))) / 2,
                ''
            );
        } else if (
            self.typeoftoken[p.ta2] == TokenUtilsLib.TypeToken.typeERC721
        ) {
            IERC721(p.ta1).transferFrom(
                self
                .tradepairs[p.ta1][p.ti1][p.ta2][p.ti2].buybids.nodes[bn].owner,
                self
                .tradepairs[p.ta1][p.ti1][p.ta2][p.ti2]
                    .sellbids
                    .nodes[sn]
                    .owner,
                p.ti1
            );
        } else {
            revert();
        }

        emit eventSingleMatch(bn, sn, amnt);
        return (true);
    }

    function buyMaxSellMinPricesAmounts(
        LibBourseStorage.BourseData storage self,
        address ta1,
        uint ti1,
        address ta2,
        uint ti2
    ) public view returns (uint, uint, uint, uint) {
        uint bn;
        uint sn;

        if (existsTradePair(self, ta1, ti1, ta2, ti2)) {
            bn = self.tradepairs[ta1][ti1][ta2][ti2].buybids.maxPriceNode(
                self.tradepairs[ta1][ti1][ta2][ti2].buybids.treeroot
            );
            sn = self.tradepairs[ta1][ti1][ta2][ti2].sellbids.minPriceNode(
                self.tradepairs[ta1][ti1][ta2][ti2].sellbids.treeroot
            );
            return (
                self.tradepairs[ta1][ti1][ta2][ti2].buybids.nodes[bn].amount,
                (self.tradepairs[ta1][ti1][ta2][ti2].buybids.nodes[bn].price >>
                    self.COUNTERBITS),
                self.tradepairs[ta1][ti1][ta2][ti2].sellbids.nodes[sn].amount,
                (self.tradepairs[ta1][ti1][ta2][ti2].sellbids.nodes[sn].price >>
                    self.COUNTERBITS)
            );
        } else return (0, 0, 0, 0);
    }

    function existsSingleMatch(
        LibBourseStorage.BourseData storage self,
        address ta1,
        uint ti1,
        address ta2,
        uint ti2
    ) public view returns (bool) {
        uint buymaxprice = self
        .tradepairs[ta1][ti1][ta2][ti2].buybids.maxPrice() >> self.COUNTERBITS;
        uint sellminprice = self
        .tradepairs[ta1][ti1][ta2][ti2].sellbids.minPrice() >> self.COUNTERBITS;
        return (buymaxprice >= sellminprice);
    }

    function getIthTradePair(
        LibBourseStorage.BourseData storage self,
        uint i
    ) public view returns (address, uint, address, uint) {
        if (i < self.tokenpairlist.length)
            return (
                self.tokenpairlist[i].ta1,
                self.tokenpairlist[i].ti1,
                self.tokenpairlist[i].ta2,
                self.tokenpairlist[i].ti2
            );
        else return (address(0x0), 0, address(0x0), 0);
    }

    function registerTradePair(
        LibBourseStorage.BourseData storage self,
        address ta1,
        uint ti1,
        address ta2,
        uint ti2
    ) public returns (bool) {
        require(allowedCurrencyCoin(self, ta2, ti2));
        if (!existsTradePair(self, ta1, ti1, ta2, ti2)) {
            if (self.typeoftoken[ta1] == TokenUtilsLib.TypeToken(0))
                self.typeoftoken[ta1] = TokenUtilsLib.getTokenType(ta1);
            self.tradepairs[ta1][ti1][ta2][ti2].buybids.AVLTreeLibInit();
            self.tradepairs[ta1][ti1][ta2][ti2].sellbids.AVLTreeLibInit();
            self.tradepairs[ta1][ti1][ta2][ti2].numbuybids = 0;
            self.tradepairs[ta1][ti1][ta2][ti2].numsellbids = 0;
            self.tradepairs[ta1][ti1][ta2][ti2].volbuybids = 0;
            self.tradepairs[ta1][ti1][ta2][ti2].volsellbids = 0;
            self.tradepairs[ta1][ti1][ta2][ti2].ascendcounter = 1;
            self.tradepairs[ta1][ti1][ta2][ti2].descendcounter =
                2 ** self.COUNTERBITS -
                1;
            self.tradepairs[ta1][ti1][ta2][ti2].tp.ta1 = ta1;
            self.tradepairs[ta1][ti1][ta2][ti2].tp.ta2 = ta2;
            self.tradepairs[ta1][ti1][ta2][ti2].tp.ti1 = ti1;
            self.tradepairs[ta1][ti1][ta2][ti2].tp.ti2 = ti2;
            self.tokenpairlist.push(self.tradepairs[ta1][ti1][ta2][ti2].tp);
            return (true);
        }
        return (false);
    }
}

///////////////////////// FAME Data Bourse Facet contract /////////////////
contract BourseFacet {
    /////////////////////// Types
    using AVLTreeLib for AVLTreeLib.AVLTree;
    using BourseLib for LibBourseStorage.BourseData;

    /////////////////////// State variables kept in struct and passed to libraries as self (first arg)
    // BourseLib.BourseData boursedata;

    /////////////////////// Modifiers

    //modifier notBlocked() {
    //   require(notBlockedInsufficientAllowance(msg.sender),"Blocked account") ;
    //   _;
    //}

    //modifier allowedCurrency(address ta2,uint ti2) {
    //   require(boursedata.allowedCurrencyCoin(ta2,ti2),"Not allowed currency") ;
    //   _;
    //}

    //modifier existingPair(address ta1,uint ti1,address ta2,uint ti2) {
    //   require(boursedata.existsTradePair(ta1,ti1,ta2,ti2),"Non-existent trade pair" ) ;
    //   _;
    //}

    /////////////////////// Events
    event eventBuyLimitOrder(address indexed useraddr, uint buybidno);
    event eventSellLimitOrder(address indexed useraddr, uint sellbidno);
    event eventCancelBuyLimitOrder(address indexed useraddr, uint buybidno);
    event eventCancelSellLimitOrder(address indexed useraddr, uint sellbidno);
    event eventDirectSell(
        address ta1,
        uint indexed ti1,
        address ta2,
        uint ti2,
        uint saleTxNo
    );

    ////////////////////// Functions
    function initialize(address escrowAddress) public {
        LibBourseStorage.BourseData storage boursedata = LibBourseStorage
            .bourseStorage();
        boursedata.escrowaddr = escrowAddress;
        boursedata.COUNTERBITS = 48; // used to implement first come first served in bid processing
        boursedata.blockedduration = 60 * 60 seconds;
    }

    fallback() external {
        revert();
    }

    ///////// Only owner functions

    function allowCurrencyCoin(address ta, uint ti) public {
        LibDiamond.enforceIsContractOwner();
        LibBourseStorage.BourseData storage boursedata = LibBourseStorage
            .bourseStorage();

        boursedata.typeoftoken[ta] = TokenUtilsLib.getTokenType(ta);
        boursedata.currencycoin[ta][ti] = true;
    }
    //////////

    function registerTradePair(
        address ta1,
        uint ti1,
        address ta2,
        uint ti2
    ) public returns (bool) {
        LibBourseStorage.BourseData storage boursedata = LibBourseStorage
            .bourseStorage();

        return (boursedata.registerTradePair(ta1, ti1, ta2, ti2));
    }

    // note that:  msg.sender : is the calling contract address who minted and owns the token
    //             seller :  is the seller address that will get the money
    //             buyer :  is the buyer address
    function directSell(
        address seller,
        address buyer,
        address ta1,
        uint ti1,
        address ta2,
        uint ti2,
        uint amnt,
        uint saleprice
    ) public {
        LibBourseStorage.BourseData storage boursedata = LibBourseStorage
            .bourseStorage();

        LibBourseStorage.TokenPair memory p = LibBourseStorage.TokenPair(
            ta1,
            ti1,
            ta2,
            ti2
        );

        require(notBlockedInsufficientAllowance(msg.sender), 'Blocked account');
        require(
            boursedata.allowedCurrencyCoin(ta2, ti2),
            'Not allowed currency'
        );
        // Note token pair registration can be done separately
        //require(boursedata.existsTradePair(ta1,ti1,ta2,ti2),"Non-existent trade pair" ) ;
        if (!boursedata.existsTradePair(ta1, ti1, ta2, ti2)) {
            registerTradePair(ta1, ti1, ta2, ti2);
        }
        if (boursedata.typeoftoken[ta1] == TokenUtilsLib.TypeToken.typeERC721)
            require(amnt == 1);
        require(
            TokenUtilsLib.checkAllowance(
                boursedata.typeoftoken[ta1],
                ta1,
                ti1,
                msg.sender,
                amnt
            ),
            'Insufficient approved allowance given to Bourse by data provider contract'
        );
        require(
            TokenUtilsLib.checkAllowance(
                boursedata.typeoftoken[ta2],
                ta2,
                ti2,
                buyer,
                amnt * saleprice
            ),
            'Insufficient approved allowance given to Bourse by the user'
        );

        recordPayNode(
            seller,
            buyer,
            ta1,
            ti1,
            ta2,
            ti2,
            amnt,
            saleprice,
            uint8(LibBourseStorage.PayTypeStatus.DIRECTSALE)
        );
        emit eventDirectSell(
            ta1,
            ti1,
            ta2,
            ti2,
            boursedata.tradepairs[ta1][ti1][ta2][ti2].sellbids.nodes.length - 1
        );

        TokenUtilsLib.tokenTransfers(
            boursedata,
            seller,
            buyer,
            p,
            amnt,
            saleprice
        );
    }

    function recordPayNode(
        address seller,
        address buyer,
        address ta1,
        uint ti1,
        address ta2,
        uint ti2,
        uint amnt,
        uint saleprice,
        uint8 paytype
    ) public {
        LibBourseStorage.BourseData storage boursedata = LibBourseStorage
            .bourseStorage();
        uint augsaleprice; // augmented offer price - augmented(padded) with 48 bit counter value

        uint btime = block.timestamp;
        augsaleprice =
            (saleprice << boursedata.COUNTERBITS) +
            boursedata.tradepairs[ta1][ti1][ta2][ti2].ascendcounter;
        boursedata.tradepairs[ta1][ti1][ta2][ti2].ascendcounter++;
        boursedata.tradepairs[ta1][ti1][ta2][ti2].sellbids.nodes.push(
            AVLTreeLib.Node({
                price: augsaleprice,
                amount: amnt,
                owner: seller,
                left: 0,
                right: 0,
                height: 1,
                splitprev: 0,
                txtime: btime,
                status: paytype
            })
        );
        boursedata
        .tradepairs[ta1][ti1][ta2][ti2].sellbids.ownerbidlist[seller].push(
                boursedata
                .tradepairs[ta1][ti1][ta2][ti2].sellbids.nodes.length - 1
            );

        augsaleprice =
            (saleprice << boursedata.COUNTERBITS) +
            boursedata.tradepairs[ta1][ti1][ta2][ti2].descendcounter;
        boursedata.tradepairs[ta1][ti1][ta2][ti2].descendcounter--;
        boursedata.tradepairs[ta1][ti1][ta2][ti2].buybids.nodes.push(
            AVLTreeLib.Node({
                price: augsaleprice,
                amount: amnt,
                owner: buyer,
                left: 0,
                right: 0,
                height: 1,
                splitprev: 0,
                txtime: btime,
                status: paytype
            })
        );
        boursedata
        .tradepairs[ta1][ti1][ta2][ti2].buybids.ownerbidlist[buyer].push(
                boursedata.tradepairs[ta1][ti1][ta2][ti2].buybids.nodes.length -
                    1
            );

        boursedata.tradepairs[ta1][ti1][ta2][ti2].numsellbids++;
        boursedata.tradepairs[ta1][ti1][ta2][ti2].numbuybids++;
        boursedata.tradepairs[ta1][ti1][ta2][ti2].volsellbids += amnt;
        boursedata.tradepairs[ta1][ti1][ta2][ti2].volbuybids += amnt;
    }

    // note that:  msg.sender : is the calling contract address who minted and owns the token
    //             seller :  is the seller address that will get the money
    //             buyer :  is the buyer address
    function directSellPAYG(
        address seller,
        address buyer,
        address ta1,
        uint ti1,
        address ta2,
        uint ti2,
        uint amnt,
        uint saleprice
    ) public {
        LibBourseStorage.BourseData storage boursedata = LibBourseStorage
            .bourseStorage();

        ESCROW(boursedata.escrowaddr).deposit(buyer, amnt * saleprice, ti1);
        recordPayNode(
            seller,
            buyer,
            ta1,
            ti1,
            ta2,
            ti2,
            amnt,
            saleprice,
            uint8(LibBourseStorage.PayTypeStatus.PAY_AS_YOU_GO)
        );
        emit eventDirectSell(
            ta1,
            ti1,
            ta2,
            ti2,
            boursedata.tradepairs[ta1][ti1][ta2][ti2].sellbids.nodes.length - 1
        );
    }

    function directSellPAYU(
        address seller,
        address buyer,
        address ta1,
        uint ti1,
        address ta2,
        uint ti2,
        uint amnt,
        uint saleprice
    ) public {
        LibBourseStorage.BourseData storage boursedata = LibBourseStorage
            .bourseStorage();

        ESCROW(boursedata.escrowaddr).deposit(buyer, amnt * saleprice, ti1);
        recordPayNode(
            seller,
            buyer,
            ta1,
            ti1,
            ta2,
            ti2,
            amnt,
            saleprice,
            uint8(LibBourseStorage.PayTypeStatus.PAY_AS_YOU_USE)
        );
        emit eventDirectSell(
            ta1,
            ti1,
            ta2,
            ti2,
            boursedata.tradepairs[ta1][ti1][ta2][ti2].sellbids.nodes.length - 1
        );
    }

    function buyLimitOrder(
        address ta1,
        uint ti1,
        address ta2,
        uint ti2,
        uint amnt,
        uint offerprice
    ) public {
        LibBourseStorage.BourseData storage boursedata = LibBourseStorage
            .bourseStorage();

        uint augofferprice; // augmented offer price - augmented(padded) with 48 bit counter value

        require(notBlockedInsufficientAllowance(msg.sender), 'Blocked account');
        require(
            boursedata.allowedCurrencyCoin(ta2, ti2),
            'Not allowed currency'
        );
        require(
            boursedata.existsTradePair(ta1, ti1, ta2, ti2),
            'Non-existent trade pair'
        );
        if (boursedata.typeoftoken[ta1] == TokenUtilsLib.TypeToken.typeERC721)
            require(amnt == 1);
        require(
            TokenUtilsLib.checkAllowance(
                boursedata.typeoftoken[ta2],
                ta2,
                ti2,
                msg.sender,
                amnt * offerprice
            ),
            'Insufficient allowance'
        );

        augofferprice =
            (offerprice << boursedata.COUNTERBITS) +
            boursedata.tradepairs[ta1][ti1][ta2][ti2].descendcounter;
        boursedata.tradepairs[ta1][ti1][ta2][ti2].descendcounter--;
        boursedata.tradepairs[ta1][ti1][ta2][ti2].buybids.treeInsertNode(
            augofferprice,
            amnt,
            msg.sender,
            0
        );
        boursedata.tradepairs[ta1][ti1][ta2][ti2].numbuybids++;
        boursedata.tradepairs[ta1][ti1][ta2][ti2].volbuybids += amnt;
        emit eventBuyLimitOrder(
            msg.sender,
            boursedata.tradepairs[ta1][ti1][ta2][ti2].buybids.nodes.length - 1
        );
    }

    function sellLimitOrder(
        address ta1,
        uint ti1,
        address ta2,
        uint ti2,
        uint amnt,
        uint offerprice
    ) public {
        LibBourseStorage.BourseData storage boursedata = LibBourseStorage
            .bourseStorage();

        uint augofferprice; // augmented offer price - augmented(padded) with 48 bit counter value

        require(notBlockedInsufficientAllowance(msg.sender), 'Blocked account');
        require(
            boursedata.allowedCurrencyCoin(ta2, ti2),
            'Not allowed currency'
        );
        require(
            boursedata.existsTradePair(ta1, ti1, ta2, ti2),
            'Non-existent trade pair'
        );
        if (boursedata.typeoftoken[ta1] == TokenUtilsLib.TypeToken.typeERC721)
            require(amnt == 1);
        require(
            TokenUtilsLib.checkAllowance(
                boursedata.typeoftoken[ta1],
                ta1,
                ti1,
                msg.sender,
                amnt
            ),
            'Insufficient allowance'
        );

        augofferprice =
            (offerprice << boursedata.COUNTERBITS) +
            boursedata.tradepairs[ta1][ti1][ta2][ti2].ascendcounter;
        boursedata.tradepairs[ta1][ti1][ta2][ti2].ascendcounter++;
        boursedata.tradepairs[ta1][ti1][ta2][ti2].sellbids.treeInsertNode(
            augofferprice,
            amnt,
            msg.sender,
            0
        );
        boursedata.tradepairs[ta1][ti1][ta2][ti2].numsellbids++;
        boursedata.tradepairs[ta1][ti1][ta2][ti2].volsellbids += amnt;
        emit eventSellLimitOrder(
            msg.sender,
            boursedata.tradepairs[ta1][ti1][ta2][ti2].sellbids.nodes.length - 1
        );
    }

    function cancelBuyLimitOrder(
        address ta1,
        uint ti1,
        address ta2,
        uint ti2,
        uint orderno
    ) public {
        LibBourseStorage.BourseData storage boursedata = LibBourseStorage
            .bourseStorage();

        require(
            boursedata.existsTradePair(ta1, ti1, ta2, ti2),
            'Non-existent trade pair'
        );
        require(
            msg.sender ==
                boursedata
                .tradepairs[ta1][ti1][ta2][ti2].buybids.nodes[orderno].owner,
            'Not bid owner'
        );
        boursedata.tradepairs[ta1][ti1][ta2][ti2].buybids.treeDeleteNode(
            boursedata
            .tradepairs[ta1][ti1][ta2][ti2].buybids.nodes[orderno].price
        );
        emit eventCancelBuyLimitOrder(msg.sender, orderno);
    }

    function cancelSellLimitOrder(
        address ta1,
        uint ti1,
        address ta2,
        uint ti2,
        uint orderno
    ) public {
        LibBourseStorage.BourseData storage boursedata = LibBourseStorage
            .bourseStorage();

        require(
            boursedata.existsTradePair(ta1, ti1, ta2, ti2),
            'Non-existent trade pair'
        );
        require(
            msg.sender ==
                boursedata
                .tradepairs[ta1][ti1][ta2][ti2].sellbids.nodes[orderno].owner,
            'Not bid owner'
        );
        boursedata.tradepairs[ta1][ti1][ta2][ti2].sellbids.treeDeleteNode(
            boursedata
            .tradepairs[ta1][ti1][ta2][ti2].sellbids.nodes[orderno].price
        );
        emit eventCancelSellLimitOrder(msg.sender, orderno);
    }

    function existsSingleMatch(
        address ta1,
        uint ti1,
        address ta2,
        uint ti2
    ) public view returns (bool) {
        LibBourseStorage.BourseData storage boursedata = LibBourseStorage
            .bourseStorage();
        return (boursedata.existsSingleMatch(ta1, ti1, ta2, ti2));
    }

    function buyMaxSellMinPricesAmounts(
        address ta1,
        uint ti1,
        address ta2,
        uint ti2
    ) public view returns (uint, uint, uint, uint) {
        LibBourseStorage.BourseData storage boursedata = LibBourseStorage
            .bourseStorage();
        return (boursedata.buyMaxSellMinPricesAmounts(ta1, ti1, ta2, ti2));
    }

    function singleMatch2(address ta1, uint ti1, address ta2, uint ti2) public {
        LibBourseStorage.TokenPair memory p = LibBourseStorage.TokenPair(
            ta1,
            ti1,
            ta2,
            ti2
        );

        singleMatch(p);
    }

    function singleMatch(
        LibBourseStorage.TokenPair memory p
    ) public returns (bool) {
        LibBourseStorage.BourseData storage boursedata = LibBourseStorage
            .bourseStorage();
        return (boursedata.singleMatch(p));
    }

    function notBlockedInsufficientAllowance(
        address useraddr
    ) public view returns (bool) {
        LibBourseStorage.BourseData storage boursedata = LibBourseStorage
            .bourseStorage();
        return ((block.timestamp - boursedata.allowanceblocked[useraddr]) >
            boursedata.blockedduration);
    }

    function setBlockedDuration(uint t) public {
        LibDiamond.enforceIsContractOwner();
        LibBourseStorage.BourseData storage boursedata = LibBourseStorage
            .bourseStorage();
        boursedata.blockedduration = t * (1 seconds);
    }

    function getNumberOfTradePairs() public view returns (uint) {
        LibBourseStorage.BourseData storage boursedata = LibBourseStorage
            .bourseStorage();
        return (boursedata.tokenpairlist.length);
    }

    function getIthTradePair(
        uint i
    ) public view returns (address, uint, address, uint) {
        LibBourseStorage.BourseData storage boursedata = LibBourseStorage
            .bourseStorage();
        return (
            boursedata.tokenpairlist[i].ta1,
            boursedata.tokenpairlist[i].ti1,
            boursedata.tokenpairlist[i].ta2,
            boursedata.tokenpairlist[i].ti2
        );
    }

    function getTotalNumVolBuyBids(
        address ta1,
        uint ti1,
        address ta2,
        uint ti2
    ) public view returns (uint, uint) {
        LibBourseStorage.BourseData storage boursedata = LibBourseStorage
            .bourseStorage();
        if (boursedata.existsTradePair(ta1, ti1, ta2, ti2))
            return (
                boursedata.tradepairs[ta1][ti1][ta2][ti2].numbuybids,
                boursedata.tradepairs[ta1][ti1][ta2][ti2].volbuybids
            );
        else return (0, 0);
    }

    function getTotalNumVolSellBids(
        address ta1,
        uint ti1,
        address ta2,
        uint ti2
    ) public view returns (uint, uint) {
        LibBourseStorage.BourseData storage boursedata = LibBourseStorage
            .bourseStorage();
        if (boursedata.existsTradePair(ta1, ti1, ta2, ti2))
            return (
                boursedata.tradepairs[ta1][ti1][ta2][ti2].numsellbids,
                boursedata.tradepairs[ta1][ti1][ta2][ti2].volsellbids
            );
        else return (0, 0);
    }

    function getNumberOfBuyBids(
        address ta1,
        uint ti1,
        address ta2,
        uint ti2
    ) public view returns (uint) {
        LibBourseStorage.BourseData storage boursedata = LibBourseStorage
            .bourseStorage();
        return (boursedata.tradepairs[ta1][ti1][ta2][ti2].buybids.nodes.length -
            1);
    }

    function getNumberOfSellBids(
        address ta1,
        uint ti1,
        address ta2,
        uint ti2
    ) public view returns (uint) {
        LibBourseStorage.BourseData storage boursedata = LibBourseStorage
            .bourseStorage();
        return (boursedata
        .tradepairs[ta1][ti1][ta2][ti2].sellbids.nodes.length - 1);
    }

    function getIthBuyBid(
        uint i,
        address ta1,
        uint ti1,
        address ta2,
        uint ti2
    ) public view returns (uint, uint, uint, uint, uint) {
        LibBourseStorage.BourseData storage boursedata = LibBourseStorage
            .bourseStorage();
        if (
            i < boursedata.tradepairs[ta1][ti1][ta2][ti2].buybids.nodes.length
        ) {
            return (
                boursedata
                .tradepairs[ta1][ti1][ta2][ti2].buybids.nodes[i].price >>
                    boursedata.COUNTERBITS,
                boursedata
                .tradepairs[ta1][ti1][ta2][ti2].buybids.nodes[i].amount,
                boursedata
                .tradepairs[ta1][ti1][ta2][ti2].buybids.nodes[i].txtime,
                boursedata
                .tradepairs[ta1][ti1][ta2][ti2].buybids.nodes[i].splitprev,
                boursedata
                .tradepairs[ta1][ti1][ta2][ti2].buybids.nodes[i].status
            );
        } else return (0, 0, 0, 0, 0);
    }

    function getIthSellBid(
        uint i,
        address ta1,
        uint ti1,
        address ta2,
        uint ti2
    ) public view returns (uint, uint, uint, uint, uint) {
        LibBourseStorage.BourseData storage boursedata = LibBourseStorage
            .bourseStorage();
        if (
            i < boursedata.tradepairs[ta1][ti1][ta2][ti2].sellbids.nodes.length
        ) {
            return (
                boursedata
                .tradepairs[ta1][ti1][ta2][ti2].sellbids.nodes[i].price >>
                    boursedata.COUNTERBITS,
                boursedata
                .tradepairs[ta1][ti1][ta2][ti2].sellbids.nodes[i].amount,
                boursedata
                .tradepairs[ta1][ti1][ta2][ti2].sellbids.nodes[i].txtime,
                boursedata
                .tradepairs[ta1][ti1][ta2][ti2].sellbids.nodes[i].splitprev,
                boursedata
                .tradepairs[ta1][ti1][ta2][ti2].sellbids.nodes[i].status
            );
        } else return (0, 0, 0, 0, 0);
    }

    function getNumberOfTraderBuyBids(
        address trader,
        address ta1,
        uint ti1,
        address ta2,
        uint ti2
    ) public view returns (uint) {
        LibBourseStorage.BourseData storage boursedata = LibBourseStorage
            .bourseStorage();
        return (
            boursedata
            .tradepairs[ta1][ti1][ta2][ti2].buybids.ownerbidlist[trader].length
        );
    }

    function getIthTraderBuyBid(
        uint i,
        address trader,
        address ta1,
        uint ti1,
        address ta2,
        uint ti2
    ) public view returns (uint, uint, uint, uint) {
        LibBourseStorage.BourseData storage boursedata = LibBourseStorage
            .bourseStorage();
        if (
            i <
            boursedata
            .tradepairs[ta1][ti1][ta2][ti2].buybids.ownerbidlist[trader].length
        ) {
            uint orderno = boursedata
            .tradepairs[ta1][ti1][ta2][ti2].buybids.ownerbidlist[trader][i];
            return (
                orderno,
                boursedata
                .tradepairs[ta1][ti1][ta2][ti2].buybids.nodes[orderno].price >>
                    boursedata.COUNTERBITS,
                boursedata
                .tradepairs[ta1][ti1][ta2][ti2].buybids.nodes[orderno].amount,
                boursedata
                .tradepairs[ta1][ti1][ta2][ti2].buybids.nodes[orderno].txtime
                // commented out due to stack too deep
                // boursedata
                // .tradepairs[ta1][ti1][ta2][ti2].buybids.nodes[orderno].splitprev,
                //boursedata
                //.tradepairs[ta1][ti1][ta2][ti2].buybids.nodes[orderno].status
            );
        } else return (0, 0, 0, 0);
    }

    function getNumberOfTraderSellBids(
        address trader,
        address ta1,
        uint ti1,
        address ta2,
        uint ti2
    ) public view returns (uint) {
        LibBourseStorage.BourseData storage boursedata = LibBourseStorage
            .bourseStorage();
        return (
            boursedata
            .tradepairs[ta1][ti1][ta2][ti2].sellbids.ownerbidlist[trader].length
        );
    }

    function getIthTraderSellBid(
        uint i,
        address trader,
        address ta1,
        uint ti1,
        address ta2,
        uint ti2
    ) public view returns (uint, uint, uint, uint) {
        LibBourseStorage.BourseData storage boursedata = LibBourseStorage
            .bourseStorage();
        if (
            i <
            boursedata
            .tradepairs[ta1][ti1][ta2][ti2].sellbids.ownerbidlist[trader].length
        ) {
            uint orderno = boursedata
            .tradepairs[ta1][ti1][ta2][ti2].buybids.ownerbidlist[trader][i];
            return (
                orderno,
                boursedata
                .tradepairs[ta1][ti1][ta2][ti2].sellbids.nodes[orderno].price >>
                    boursedata.COUNTERBITS,
                boursedata
                .tradepairs[ta1][ti1][ta2][ti2].sellbids.nodes[orderno].amount,
                boursedata
                .tradepairs[ta1][ti1][ta2][ti2].sellbids.nodes[orderno].txtime
                // commented out due to stack too deep
                // boursedata
                // .tradepairs[ta1][ti1][ta2][ti2].sellbids.nodes[orderno].splitprev,
                //boursedata
                //.tradepairs[ta1][ti1][ta2][ti2].sellbids.nodes[orderno].status
            );
        } else return (0, 0, 0, 0);
    }

    function getTokenType(address ta) public view returns (uint) {
        LibBourseStorage.BourseData storage boursedata = LibBourseStorage
            .bourseStorage();
        return (uint(boursedata.typeoftoken[ta]));
    }

    function supportsInterface(
        bytes4 interfaceID
    ) external pure returns (bool) {
        return ((interfaceID == this.supportsInterface.selector) || // ERC165
            (interfaceID ==
                this.sellLimitOrder.selector ^
                    this.buyLimitOrder.selector ^
                    this.directSell.selector ^
                    this.cancelSellLimitOrder.selector ^
                    this.singleMatch2.selector ^
                    this.singleMatch.selector ^
                    this.buyMaxSellMinPricesAmounts.selector ^
                    this.cancelBuyLimitOrder.selector));
    }
}
