// SPDX-License-Identifier: MIT
pragma solidity ^0.8.17;

import './PaymentToken.sol';
import './DataAccessPayAsYouUse.sol';
import './OfferingToken.sol';
import '@openzeppelin/contracts/access/Ownable.sol';

contract Escrow is Ownable {
    PaymentToken paymentToken;
    OfferingToken offeringToken;
    address bourse;

    mapping(address => mapping(uint256 => uint256)) public consumerFundsPerId;

    event AssetUsed(address buyer, uint256 assetId, uint256 amount);
    event Deposit(address account, uint256 amount);
    event Withdraw(address account, uint256 amount);

    constructor(address _tokenAddress) {
        paymentToken = PaymentToken(_tokenAddress);
    }

    function setBourse(address bourseAddress) external onlyOwner {
        bourse = bourseAddress;
    }

    function getBourse() external view returns (address) {
        return address(bourse);
    }

    modifier onlyBourse() {
        require(
            msg.sender == address(bourse),
            'Can only be called by the bourse contract!'
        );
        _;
    }

    function useAsset(
        address user,
        uint256 id,
        uint256 amount
    ) external onlyOwner {
        address dataProvider = offeringToken.getOfferingTokenIndxDataProvider(
            id
        );

        require(
            dataProvider != address(0),
            'Asset with this ID does not exist'
        );
        require(amount > 0, 'Amount must be greater than zero');

        consumerFundsPerId[user][id] -= amount;
        paymentToken.transferFrom(address(this), dataProvider, amount);
        emit AssetUsed(msg.sender, id, amount);
    }

    function deposit(
        address user,
        uint256 amount,
        uint256 id
    ) external onlyBourse {
        require(amount > 0, 'Amount must be greater than zero!');

        consumerFundsPerId[user][id] += amount;
        paymentToken.transferFrom(user, address(this), amount);
        emit Deposit(user, amount);
    }

    function withdraw(uint256 amount, uint256 id) external {
        require(amount > 0, 'Amount must be greater than zero!');
        require(
            consumerFundsPerId[msg.sender][id] >= amount,
            'Not enough consumer funds deposited!'
        );

        consumerFundsPerId[msg.sender][id] -= amount;
        paymentToken.transferFrom(address(this), msg.sender, amount);
        emit Withdraw(owner(), amount);
    }
}
