pragma solidity ^0.8.0;

// SPDX-License-Identifier: Apache-2

import '@openzeppelin/contracts/utils/introspection/ERC165.sol';
import './Bourse.sol';

import './diamond/Diamond.sol';

interface ERC20 {
    // PaymentToken  with added burn and mint functions
    function balanceOf(address owner_addr) external view returns (uint256);
    function transfer(address to_addr, uint256 amnt) external returns (bool);
    function transferFrom(
        address from_addr,
        address to_addr,
        uint256 amnt
    ) external returns (bool);
    function allowance(
        address owner_addr,
        address spender_addr
    ) external view returns (uint256);
    function approve(address _spender, uint256 _value) external returns (bool);
    function burn(address from, uint256 amount, bytes memory _data) external;
    function mint(address to, uint256 amount, bytes memory _data) external;
    function setPermissionContract(address _permissionContract) external;
}

interface ERC1155 {
    function balanceOf(
        address owner_addr,
        uint256 token_id
    ) external view returns (uint256);
}

interface IBourse {
    function allowCurrencyCoin(address cointokenaddr, uint tokindx) external;
}

library LibGovernanceStorage {
    bytes32 constant GOVERNANCE_STORAGE_POSITION =
        keccak256('diamond.storage.GovernanceStorage.v1');

    struct Token {
        // both ERC20 and ERC1155
        address ta; // token contract address
        uint ti; // token index (for ERC20, put this as 0)
    }

    struct GovernanceData {
        Token[] coins; // addresss, indx array for  coin contracts
        address famebourse; // address Fame Bourse contract
        mapping(address => mapping(uint => string)) coin2symbol;
        mapping(string => uint) coinsym2coinno;
        mapping(address => uint) regdataproviders; // data provider expiration datetime
    }

    function governanceStorage()
        internal
        pure
        returns (GovernanceData storage gd)
    {
        bytes32 position = GOVERNANCE_STORAGE_POSITION;
        // solhint-disable-next-line no-inline-assembly
        assembly {
            gd.slot := position
        }
    }
}

contract GovernanceFacet {
    function initialize() public {
        LibDiamond.enforceIsContractOwner();
        LibGovernanceStorage.GovernanceData
            storage govdata = LibGovernanceStorage.governanceStorage();
        if (govdata.coins.length == 0) {
            govdata.coins.push(LibGovernanceStorage.Token(address(0x0), 0)); //dummy ,not used, indices start with 1
        }
    }

    function setFameBourse(address addr) public {
        LibDiamond.enforceIsContractOwner();
        LibGovernanceStorage.GovernanceData
            storage govdata = LibGovernanceStorage.governanceStorage();
        govdata.famebourse = addr;
    }

    function getFameBourse() public view returns (address) {
        LibGovernanceStorage.GovernanceData
            storage govdata = LibGovernanceStorage.governanceStorage();
        return (govdata.famebourse);
    }

    function registerCoinToken(
        address cointokenaddr,
        uint tokindx,
        string memory coinsymbol
    ) public returns (uint) {
        LibDiamond.enforceIsContractOwner();
        LibGovernanceStorage.GovernanceData
            storage govdata = LibGovernanceStorage.governanceStorage();
        require(govdata.coinsym2coinno[coinsymbol] == 0);
        govdata.coins.push(LibGovernanceStorage.Token(cointokenaddr, tokindx));
        govdata.coin2symbol[cointokenaddr][tokindx] = coinsymbol;
        govdata.coinsym2coinno[coinsymbol] = govdata.coins.length - 1;
        IBourse(govdata.famebourse).allowCurrencyCoin(cointokenaddr, tokindx);
        return (govdata.coins.length - 1); // return coin number (numbers are 1,2,3... etc. )
    }

    function mintCoin(
        address to_tid,
        string memory coinsymbol,
        uint amount
    ) public returns (bool) {
        LibDiamond.enforceIsContractOwner();
        LibGovernanceStorage.GovernanceData
            storage govdata = LibGovernanceStorage.governanceStorage();
        uint coinno = govdata.coinsym2coinno[coinsymbol];
        require(coinno > 0, 'Coin is not registered.');
        ERC20(govdata.coins[coinno].ta).mint(to_tid, amount, '');
        return (true);
    }

    function setPaymentTokenPermissions(
        string memory coinsymbol,
        address newPermissionsAddress
    ) public {
        // 1. Ensure only the owner of the Diamond can call this
        LibDiamond.enforceIsContractOwner();

        // 2. Get the governance storage
        LibGovernanceStorage.GovernanceData
            storage govdata = LibGovernanceStorage.governanceStorage();

        // 3. Look up the PaymentToken's address by its symbol
        uint coinno = govdata.coinsym2coinno[coinsymbol];
        require(coinno > 0, 'Coin is not registered.');

        // 4. Call the setPermissionContract function on the PaymentToken contract
        ERC20(govdata.coins[coinno].ta).setPermissionContract(
            newPermissionsAddress
        );
    }

    function burnCoin(
        address from_tid,
        string memory coinsymbol,
        uint amount
    ) public returns (bool) {
        LibDiamond.enforceIsContractOwner();
        LibGovernanceStorage.GovernanceData
            storage govdata = LibGovernanceStorage.governanceStorage();
        uint coinno = govdata.coinsym2coinno[coinsymbol];
        require(coinno > 0, 'Coin is not registered.');
        ERC20(govdata.coins[coinno].ta).burn(from_tid, amount, '');
        return (true);
    }

    function getNoofCoins() public view returns (uint) {
        LibGovernanceStorage.GovernanceData
            storage govdata = LibGovernanceStorage.governanceStorage();
        return (govdata.coins.length - 1);
    }

    function getCoinBalance(
        address traderid,
        string memory tokensymbol
    ) public view returns (uint) {
        (address tokaddr, uint tokindx) = getCoinAddress(tokensymbol);
        if (getTokenStandard(tokaddr) == 1155) {
            return (ERC1155(tokaddr).balanceOf(traderid, tokindx));
        } else {
            return (ERC20(tokaddr).balanceOf(traderid));
        }
    }

    function getCoinNoSymbol(uint coinno) public view returns (string memory) {
        LibGovernanceStorage.GovernanceData
            storage govdata = LibGovernanceStorage.governanceStorage();
        require(coinno < govdata.coins.length, 'Invalid coin no.');
        LibGovernanceStorage.Token memory t = govdata.coins[coinno];
        return (govdata.coin2symbol[t.ta][t.ti]);
    }

    function getCoinAddress(
        string memory coinsym
    ) public view returns (address, uint) {
        LibGovernanceStorage.GovernanceData
            storage govdata = LibGovernanceStorage.governanceStorage();
        uint i = govdata.coinsym2coinno[coinsym];
        require(i > 0, 'Coin does not exist');
        return (govdata.coins[i].ta, govdata.coins[i].ti);
    }

    function getCoinNo(
        address coinaddr,
        uint tokindx
    ) public view returns (uint) {
        LibGovernanceStorage.GovernanceData
            storage govdata = LibGovernanceStorage.governanceStorage();
        require(bytes(govdata.coin2symbol[coinaddr][tokindx]).length > 0);
        return (govdata.coinsym2coinno[govdata.coin2symbol[coinaddr][tokindx]]);
    }

    function getCoinSymbol(
        address coinaddr,
        uint tokindx
    ) public view returns (string memory) {
        LibGovernanceStorage.GovernanceData
            storage govdata = LibGovernanceStorage.governanceStorage();
        require(bytes(govdata.coin2symbol[coinaddr][tokindx]).length > 0);
        return (govdata.coin2symbol[coinaddr][tokindx]);
    }

    function supportsInterface2(
        address accaddr,
        bytes4 interfaceID
    ) public view returns (bool) {
        bytes memory data = abi.encodeWithSelector(
            ERC165.supportsInterface.selector,
            interfaceID
        );
        (bool success, bytes memory result) = accaddr.staticcall{gas: 30000}(
            data
        );
        return ((result.length >= 0x20) &&
            success &&
            abi.decode(result, (bool)));
    }

    function registerDataProvider(address dataproaddr, uint expiration) public {
        LibDiamond.enforceIsContractOwner();
        LibGovernanceStorage.GovernanceData
            storage govdata = LibGovernanceStorage.governanceStorage();
        require(block.timestamp < expiration);
        if (govdata.regdataproviders[dataproaddr] > 0) {
            require(govdata.regdataproviders[dataproaddr] < expiration); // further check
        }
        govdata.regdataproviders[dataproaddr] = expiration;
    }

    function deregisterDataProvider(address dataprovaddr) public {
        LibDiamond.enforceIsContractOwner();
        LibGovernanceStorage.GovernanceData
            storage govdata = LibGovernanceStorage.governanceStorage();
        require(govdata.regdataproviders[dataprovaddr] > block.timestamp);
        govdata.regdataproviders[dataprovaddr] = 0;
    }

    function isRegisteredDataProvider(
        address dataprovaddr
    ) public view returns (bool) {
        LibGovernanceStorage.GovernanceData
            storage govdata = LibGovernanceStorage.governanceStorage();
        return (govdata.regdataproviders[dataprovaddr] >= block.timestamp);
    }

    function getExpirationDataProvider(
        address dataprovaddr
    ) public view returns (uint) {
        LibGovernanceStorage.GovernanceData
            storage govdata = LibGovernanceStorage.governanceStorage();
        return (govdata.regdataproviders[dataprovaddr]);
    }

    function getTokenStandard(address ta) public view returns (uint) {
        bytes4 interfaceIdERC1155 = 0xd9b67a26;
        bytes4 interfaceIdERC721 = 0x80ac58cd;

        if (supportsInterface2(ta, interfaceIdERC1155)) {
            return (1155);
        } else if (supportsInterface2(ta, interfaceIdERC721)) {
            return (721);
        }
        // otherwise assume it is ERC20
        return (20);
    }
}
