// SPDX-License-Identifier: MIT
pragma solidity ^0.8.0;

import '@openzeppelin/contracts-upgradeable/token/ERC721/ERC721Upgradeable.sol';
import '@openzeppelin/contracts-upgradeable/proxy/utils/Initializable.sol';
import '@openzeppelin/contracts-upgradeable/access/OwnableUpgradeable.sol';
import '@openzeppelin/contracts/utils/structs/EnumerableMap.sol';

contract OfferingToken is Initializable, ERC721Upgradeable, OwnableUpgradeable {
    // This struct holds all relevant information about individual offerings
    // for quick access from the token contract itself
    struct OIdInfo {
        uint256 dataAccessPrice; // Price for accessing this offering's data
        uint256 capDuration; // Duration cap for data access
        string capDownloads; // Download limit for data access
        string capVolume; // Volume limit for data access
        string cdsTarget; // Content delivery system target configuration
        string cdsSl; // Content delivery system service level configuration
        string oid; // Offering ID string for retrieval
        address dataProvider; // Data provider/beneficiary
    }

    // This struct holds all relevant information about asset, offers, beneficiaries
    // for quick access from the token contract itself
    struct AssetIdInfo {
        address dataProvider; // LEGACY Single data provider (latest one)
        uint256[] assetOidsList; // List of all offering IDs for this asset
        string assetid; // Asset ID string for retrieval
        address[] allDataProviders; // Array of all data providers
    }

    uint256 private totalMinted;
    uint256 private totalBurned;

    // oidHash => oidinfo
    mapping(uint256 => OIdInfo) private oidInfo;

    // assetidHash => list of offerings oids - existing and removed
    mapping(uint256 => AssetIdInfo) private assetInfo;

    mapping(uint256 => string) private _tokenURIs;

    // Historical offerings kept as enumerable map:
    //
    // Old code was: uint256[] private _mintedTokens;
    // In new code _mintedTokens Array is replaced with enumerable map
    // oidHash => assetidHash
    // Note that all minted offering tokens
    //  (both current + burned) are kept in the map
    // since trading history may need to do lookup on
    // oidHash => assetidHash
    // even for burned offering tokens
    using EnumerableMap for EnumerableMap.UintToUintMap;
    EnumerableMap.UintToUintMap private mintedTokens;

    event TokenOfferingMinted(
        address indexed owner,
        uint256 tokenId,
        bytes data
    );

    function initialize(
        string memory _name,
        string memory _symbol
    ) public initializer {
        __ERC721_init(_name, _symbol);
        __Ownable_init();
        totalMinted = 0;
        totalBurned = 0;
    }

    function ver() public pure virtual returns (uint8) {
        return 1;
    }

    // Existence check for currently offered tokens (burned ones do not exist)
    function tokenExists(string memory _oid) public view returns (bool) {
        return _exists(getIDHash(_oid));
    }

    // Existence check for all historically offered
    // tokens (burned ones included and historically exists)
    function historicallyTokenExists(
        string memory _oid
    ) public view returns (bool) {
        return (mintedTokens.contains(getIDHash(_oid)));
    }

    // should be addOffer, but keeping old name to avoid confusion
    function addAsset(
        string memory _assetid, // newly added
        string memory _oid,
        string memory _tokenUri,
        address dataProvider,
        uint256 _tokenOfferAmount,
        uint256 _capDuration,
        string memory _capDownloads,
        string memory _capVolume,
        string memory _cdsTarget,
        string memory _cdsSl,
        bytes memory _data
    ) external onlyOwner {
        uint256 oIdHash = getIDHash(_oid);
        uint256 assetidHash = getIDHash(_assetid);
        // saving the offering info of this oid/asset
        oidInfo[oIdHash] = OIdInfo(
            _tokenOfferAmount,
            _capDuration,
            _capDownloads,
            _capVolume,
            _cdsTarget,
            _cdsSl,
            _oid,
            dataProvider
        );
        // minting the id with given amount
        mint(_tokenUri, oIdHash, assetidHash);
        assetInfo[assetidHash].dataProvider = dataProvider; // Keep latest
        assetInfo[assetidHash].allDataProviders.push(dataProvider); // Add to array
        assetInfo[assetidHash].assetid = _assetid;
        assetInfo[assetidHash].assetOidsList.push(oIdHash);
        emit TokenOfferingMinted(dataProvider, oIdHash, _data);
    }

    function mint(
        string memory _tokenUri,
        uint256 _tokenId,
        uint256 _assetid
    ) private {
        _safeMint(msg.sender, _tokenId);
        _tokenURIs[_tokenId] = _tokenUri;
        mintedTokens.set(_tokenId, _assetid);
        totalMinted++;
    }

    function getTotalMinted() public view returns (uint256) {
        return (totalMinted);
    }

    function getTotalBurned() public view returns (uint256) {
        return (totalBurned);
    }

    function removeOffering(string memory _oid) public onlyOwner {
        uint256 tokenId = getIDHash(_oid);
        burn(tokenId);
    }

    function burn(uint256 tokenIndx) private {
        _burn(tokenIndx); // reverts if the token does not exist
        totalBurned++;
    }

    function tokenURI(
        uint256 tokenId
    ) public view virtual override returns (string memory) {
        require(_exists(tokenId), 'URI query for nonexistent token!');
        string memory _tokenURI = _tokenURIs[tokenId];

        return
            bytes(_tokenURI).length > 0
                ? string(abi.encodePacked(_tokenURI))
                : '';
    }

    function tokenURIHash(
        string memory _oid
    ) public view returns (string memory) {
        uint256 oidHash = getIDHash(_oid);

        require(_exists(oidHash), 'URI query for nonexistent token!');
        string memory _tokenURI = _tokenURIs[oidHash];

        return
            bytes(_tokenURI).length > 0
                ? string(abi.encodePacked(_tokenURI))
                : '';
    }

    // Note that these are currently existing tokens (i.e. does not include burned ones)
    // supports returning up to 2^32/32 = 2^27  elements
    function getMintedTokens() external view returns (uint256[] memory) {
        uint256 tokIndx;

        uint256 currentLength = totalMinted - totalBurned;
        uint256[] memory tokenIndxs = new uint256[](currentLength);
        uint256 j = 0;
        for (uint256 i = 0; i < mintedTokens.length(); i++) {
            (tokIndx, ) = mintedTokens.at(i);
            if (_exists(tokIndx)) {
                tokenIndxs[j] = tokIndx;
                j++;
            }
        }
        return tokenIndxs;
    }

    // Note that these are historically existing tokens (i.e. does not include burned ones)
    // supports returning up to 2^32/32 = 2^27  elements
    function getHistoricallyMintedTokens()
        external
        view
        returns (uint256[] memory)
    {
        uint256 historicalLength = mintedTokens.length();
        uint256[] memory tokenIndxs = new uint256[](historicalLength);
        for (uint256 i = 0; i < historicalLength; i++) {
            (tokenIndxs[i], ) = mintedTokens.at(i);
        }

        return tokenIndxs;
    }

    function getHistoricallyMintedTokenAt(
        uint256 i
    ) external view returns (uint256) {
        require(
            i < mintedTokens.length(),
            'Get Historically Minted Token At index out of range error'
        );
        uint tokIndx;

        (tokIndx, ) = mintedTokens.at(i);
        return tokIndx;
    }

    // Old code: the following is redundant, same functionality offered by getOfferingDataProvider
    //function getAssetOwnerDataProvider(
    //    string memory _oid
    //) external view returns (address) {
    //    return idInfo[getAssetIdFromOID(_oid)].dataProvider;
    //}

    // LEGACY
    function getAssetDataProvider(
        string memory _assetid
    ) external view returns (address) {
        return assetInfo[getIDHash(_assetid)].dataProvider;
    }

    function getAllAssetDataProviders(
        string memory _assetid
    ) external view returns (address[] memory) {
        return assetInfo[getIDHash(_assetid)].allDataProviders;
    }

    function getOfferingDataProvider(
        string memory _oid
    ) external view returns (address) {
        uint256 oidHash = getIDHash(_oid);
        return oidInfo[oidHash].dataProvider;
    }

    function getOfferingTokenIndxDataProvider(
        uint256 _tokIndx
    ) external view returns (address) {
        return oidInfo[_tokIndx].dataProvider;
    }

    function getOfferDataAccessPrice(
        string memory _oid
    ) external view returns (uint256) {
        return oidInfo[getIDHash(_oid)].dataAccessPrice;
    }

    function getOfferDurationTime(
        string memory _oid
    ) external view returns (uint256) {
        return oidInfo[getIDHash(_oid)].capDuration;
    }

    function getOfferIdInfo(
        string memory _oid
    ) external view returns (OIdInfo memory _idInfo) {
        _idInfo = oidInfo[getIDHash(_oid)];
        return _idInfo;
    }

    function getOidFromHash(
        uint256 _oid
    ) external view returns (string memory) {
        string memory oid;

        oid = oidInfo[_oid].oid;
        return oid;
    }

    function getAssetidFromHash(
        uint256 _assetid
    ) external view returns (string memory) {
        string memory assetid;

        assetid = assetInfo[_assetid].assetid;
        return assetid;
    }

    // Getter function to retrieve the number of oids in the asset oid list
    function getNoOfAssetOids(
        uint256 assetidHash
    ) public view returns (uint256) {
        return assetInfo[assetidHash].assetOidsList.length;
    }

    // Getter function to retrieve  oids in the asset oid list
    function getAssetOids(
        uint256 assetidHash
    ) public view returns (uint256[] memory) {
        return assetInfo[assetidHash].assetOidsList;
    }

    // Getter function to retrieve  oid hash at a specific index in the asset's oid hash list
    function getAssetOidAtIndex(
        uint256 assetidHash,
        uint256 index
    ) public view returns (uint256) {
        require(
            index < assetInfo[assetidHash].assetOidsList.length,
            'Index out of bounds error'
        );
        return assetInfo[assetidHash].assetOidsList[index];
    }

    // Get the assetid hash corresponding to oid token index (oid string hash)
    function getAssetidHashOfOidHash(
        uint256 oidHash
    ) external view returns (uint256) {
        return (mintedTokens.get(oidHash));
    }

    // Get the assetid  corresponding to oid token index (oid string )
    function getAssetidOfOid(
        string memory oid
    ) external view returns (string memory) {
        uint256 oidHash = getIDHash(oid);
        uint256 assetidHash = mintedTokens.get(oidHash);
        return (assetInfo[assetidHash].assetid);
    }

    // The following old code has been replaced by getIDHash
    // getIDHash can be used with both assetid or oid.
    // Note that getIDHash(oid) returned hash  is used as the ERC721 token index value
    // Old code:
    //function getAssetIdFromOID(
    //    string memory _oid
    //) public pure returns (uint256) {
    //    bytes32 hash = keccak256(abi.encodePacked(_oid));
    //    return uint256(hash);
    //}
    function getIDHash(string memory _str) public pure returns (uint256) {
        bytes32 hash = keccak256(abi.encodePacked(_str));
        return uint256(hash);
    }
}
