// SPDX-License-Identifier: MIT
pragma solidity ^0.8.17;

import '@openzeppelin/contracts-upgradeable/token/ERC20/ERC20Upgradeable.sol';
import '@openzeppelin/contracts-upgradeable/proxy/utils/Initializable.sol';
import '@openzeppelin/contracts-upgradeable/access/OwnableUpgradeable.sol';
import '@openzeppelin/contracts-upgradeable/security/PausableUpgradeable.sol';

// Interface for the Permissions contract
interface IPermissionContract {
    function accountPermitted(address _account) external view returns (bool);
}

contract PaymentToken is
    Initializable,
    ERC20Upgradeable,
    OwnableUpgradeable,
    PausableUpgradeable
{
    // Permissions contract address
    address public permissionContract;

    event TokenMinted(
        address indexed owner,
        address indexed mintTo,
        uint256 mintAmount,
        bytes data
    );
    event TokenBurned(
        address indexed owner,
        address indexed burnFor,
        uint256 burnAmount,
        bytes data
    );

    event PermissionContractUpdated(
        address indexed oldContract,
        address indexed newContract
    );

    function initialize(
        string memory name,
        string memory symbol
    ) public initializer {
        __ERC20_init(name, symbol);
        __Ownable_init();
        __Pausable_init();
    }

    function ver() public pure virtual returns (uint8) {
        return 1;
    }

    // Function to set the Permissions contract address
    function setPermissionContract(
        address _permissionsContract
    ) external onlyOwner {
        address oldContract = permissionContract;
        permissionContract = _permissionsContract;
        emit PermissionContractUpdated(oldContract, _permissionsContract);
    }

    // Function to check if an account is permitted
    function isPermitted(address account) external view returns (bool) {
        if (permissionContract == address(0)) {
            return true; // If no permissions contract is set, all accounts are permitted
        }
        return
            IPermissionContract(permissionContract).accountPermitted(account);
    }

    // permission checks
    function _beforeTokenTransfer(
        address from,
        address to,
        uint256 amount
    ) internal virtual override {
        super._beforeTokenTransfer(from, to, amount);

        // Skip checks for minting (from == address(0)) and burning (to == address(0))
        if (from != address(0)) {
            // Check if sender account is permitted (if Permissions contract is set)
            if (permissionContract != address(0)) {
                require(
                    IPermissionContract(permissionContract).accountPermitted(
                        from
                    ),
                    'PaymentToken: sender account is not permitted'
                );
            }
        }
    }

    function mint(
        address to,
        uint256 amount,
        bytes memory _data
    ) public onlyOwner whenNotPaused {
        _mint(to, amount);
        emit TokenMinted(msg.sender, to, amount, _data);
    }

    function burn(
        address from,
        uint256 amount,
        bytes memory _data
    ) public onlyOwner whenNotPaused {
        _burn(from, amount);
        emit TokenBurned(msg.sender, from, amount, _data);
    }

    function transfer(
        address to,
        uint256 amount
    ) public override whenNotPaused returns (bool) {
        return ERC20Upgradeable.transfer(to, amount);
    }

    function transferFrom(
        address from,
        address to,
        uint256 amount
    ) public override whenNotPaused returns (bool) {
        return ERC20Upgradeable.transferFrom(from, to, amount);
    }

    function pause() external onlyOwner {
        _pause();
    }

    function unpause() external onlyOwner {
        _unpause();
    }
}
