// SPDX-License-Identifier: MIT
pragma solidity ^0.8.17;

// Import necessary libraries and interfaces
import './diamond/libraries/LibDiamond.sol';
import './TradingManagementStorageFacet.sol';
import './OfferingToken.sol';
import '@openzeppelin/contracts/token/ERC20/IERC20.sol';
import '@openzeppelin/contracts/token/ERC1155/IERC1155Receiver.sol';
import '@openzeppelin/contracts/utils/structs/EnumerableSet.sol';

// Define interfaces for external contracts
interface IBourse {
    function directSell(
        address from,
        address to,
        address nftContract,
        uint256 tokenId,
        address paymentToken,
        uint256 price,
        uint256 amount,
        uint256 dataAccessTokenPrice
    ) external;
}

interface IDataAccess {
    function mintAndApprove(
        uint256 _offeringTokIndx,
        address _operator,
        bytes memory _data
    ) external;
}

contract TradingManagementExecutorFacet {
    // Use the storage library
    using LibTradingManagementStorage for LibTradingManagementStorage.TradingManagementStorage;
    using EnumerableSet for EnumerableSet.UintSet;

    // Events
    event DataAccessPurchased(
        address indexed owner,
        uint256 tokenId,
        bytes data
    );
    event DataAccessBurned(address indexed owner, uint256 tokenId, bytes data);

    // Internal function to check if an asset is purchasable
    function _isAssetPurchasable(
        string memory _oid,
        LibTradingManagementStorage.TradingManagementStorage storage tms
    ) internal view {
        // Get contract addresses from storage
        address offeringTokenContract = tms.offeringTokenContract;
        OfferingToken offeringToken = OfferingToken(offeringTokenContract);
        address paymentTokenContract = tms.paymentTokenContract;
        IERC20 paymentToken = IERC20(paymentTokenContract);

        require(offeringToken.tokenExists(_oid), "This offer doesn't exist!");

        // Ensure the sender has enough payment token balance
        require(
            paymentToken.balanceOf(msg.sender) >=
                offeringToken.getOfferDataAccessPrice(_oid),
            'Insufficient payment token balance to purchase access rights!'
        );
    }

    // Internal function to handle purchase logic
    function _purchaseAccessRight(
        string memory _oid,
        bytes memory _data,
        address dataAccessContractAddress,
        LibTradingManagementStorage.TradingManagementStorage storage tms
    ) internal {
        // Get contract addresses from storage
        address offeringTokenContract = tms.offeringTokenContract;
        address bourseContractAddress = tms.bourseContractAddress;
        address paymentTokenContract = tms.paymentTokenContract;

        // Create interfaces
        OfferingToken offeringToken = OfferingToken(offeringTokenContract);
        IBourse bourse = IBourse(bourseContractAddress);
        IDataAccess dataAccessContract = IDataAccess(dataAccessContractAddress);

        uint256 offeringTokIndx = offeringToken.getIDHash(_oid);
        address dataProvider = offeringToken.getOfferingDataProvider(_oid);

        // price logic
        // Payment for SUB, PAYU tokens are transfered instantly
        // For PAYG (volume based, tms.paygDataAccessContract) payment token is withdrawn from customer by a provider after using data access token. Not instatly after purchasing it from the marketplace. The withdrawal is performed by the provider, nor any FAME contract or operator.
        uint256 price = dataAccessContractAddress == tms.paygDataAccessContract
            ? 0
            : offeringToken.getOfferDataAccessPrice(_oid);
        // Mint and approve data access token
        dataAccessContract.mintAndApprove(
            offeringTokIndx,
            bourseContractAddress,
            _data
        );

        address buyer = msg.sender;
        // Execute direct sell on Bourse contract
        bourse.directSell(
            dataProvider,
            buyer,
            dataAccessContractAddress,
            offeringTokIndx,
            address(paymentTokenContract),
            0, // Price is 0 because dataAccessTokenPrice is used
            1, // Amount
            price
        );
        // Emit event
        emit DataAccessPurchased(buyer, offeringTokIndx, _data);
    }

    // Purchase access right (Pay-As-You-Go)
    function purchaseAccessRightPAYG(
        string memory _oid,
        bytes memory _data,
        uint256 addedTime // Pass only added time
    ) external {
        LibTradingManagementStorage.TradingManagementStorage
            storage tms = LibTradingManagementStorage
                .tradingManagementStorage();

        address offeringTokenContract = tms.offeringTokenContract;
        OfferingToken offeringToken = OfferingToken(offeringTokenContract);

        _isAssetPurchasable(_oid, tms);
        address dataAccessContractAddress = tms.paygDataAccessContract;

        _purchaseAccessRight(_oid, _data, dataAccessContractAddress, tms);

        uint256 offeringTokIndx = offeringToken.getIDHash(_oid);

        uint256 expirationTime = block.timestamp + addedTime;

        // ✅ Store expiration
        tms.dataAccessExpirationPAYG[msg.sender][offeringTokIndx].push(
            expirationTime
        );
        tms.userOfferIdsPAYG[msg.sender].add(offeringTokIndx);
    }

    // Returns the expiration time in unixtime if PAYG exists
    // It will return 0, if PAYG does not exist.
    // Only the most recent PAYG time is reported (since only the most
    // recent access purchase time is kept)
    function getLastDataAccessPAYGExpirationTime(
        address _buyer,
        uint256 tokenId
    ) external view returns (uint256) {
        LibTradingManagementStorage.TradingManagementStorage
            storage tms = LibTradingManagementStorage
                .tradingManagementStorage();

        if (tms.dataAccessExpirationPAYG[_buyer][tokenId].length > 0) {
            uint lastOne = tms
            .dataAccessExpirationPAYG[_buyer][tokenId].length - 1;
            return tms.dataAccessExpirationPAYG[_buyer][tokenId][lastOne];
        } else {
            return 0;
        }
    }

    // Purchase access right (Pay-As-You-Use)
    function purchaseAccessRightPAYU(
        string memory _oid,
        bytes memory _data,
        uint256 addedTime
    ) external {
        LibTradingManagementStorage.TradingManagementStorage
            storage tms = LibTradingManagementStorage
                .tradingManagementStorage();

        address offeringTokenContract = tms.offeringTokenContract;
        OfferingToken offeringToken = OfferingToken(offeringTokenContract);

        _isAssetPurchasable(_oid, tms);
        address dataAccessContractAddress = tms.payuDataAccessContract;

        _purchaseAccessRight(_oid, _data, dataAccessContractAddress, tms);

        uint256 offeringTokIndx = offeringToken.getIDHash(_oid);

        uint256 expirationTime = block.timestamp + addedTime;

        tms.dataAccessExpirationPAYU[msg.sender][offeringTokIndx].push(
            expirationTime
        );
        tms.userOfferIdsPAYU[msg.sender].add(offeringTokIndx);
    }

    // Purchase access right (Subscription)
    function purchaseAccessRightSubscription(
        string memory _oid,
        bytes memory _data,
        uint256 addedTime // Pass only added time
    ) external {
        LibTradingManagementStorage.TradingManagementStorage
            storage tms = LibTradingManagementStorage
                .tradingManagementStorage();

        address offeringTokenContract = tms.offeringTokenContract;
        OfferingToken offeringToken = OfferingToken(offeringTokenContract);

        _isAssetPurchasable(_oid, tms);
        address dataAccessContractAddress = tms.subscriptionDataAccessContract;

        _purchaseAccessRight(_oid, _data, dataAccessContractAddress, tms);

        uint256 offeringTokIndx = offeringToken.getIDHash(_oid);

        uint256 expirationTime = block.timestamp + addedTime;

        tms.dataAccessExpirationSubscription[msg.sender][offeringTokIndx].push(
            expirationTime
        );
        tms.userOfferIdsSUB[msg.sender].add(offeringTokIndx);
    }

    /**
     * @dev Returns the entire array of expiration timestamps for PAYG purchases of a specific token by a buyer.
     */
    function getAllDataAccessExpirationsPAYG(
        address _buyer,
        uint256 tokenId
    ) external view returns (uint256[] memory) {
        LibTradingManagementStorage.TradingManagementStorage
            storage tms = LibTradingManagementStorage
                .tradingManagementStorage();
        return tms.dataAccessExpirationPAYG[_buyer][tokenId];
    }
    /**
     * @dev Returns the entire array of expiration timestamps for PAYU purchases of a specific token by a buyer.
     */
    function getAllDataAccessExpirationsPAYU(
        address _buyer,
        uint256 tokenId
    ) external view returns (uint256[] memory) {
        LibTradingManagementStorage.TradingManagementStorage
            storage tms = LibTradingManagementStorage
                .tradingManagementStorage();
        return tms.dataAccessExpirationPAYU[_buyer][tokenId];
    }

    /**
     * @dev Returns the entire array of expiration timestamps for Subscription-based purchases of a specific token by a buyer.
     */
    function getAllDataAccessExpirationsSubscription(
        address _buyer,
        uint256 tokenId
    ) external view returns (uint256[] memory) {
        LibTradingManagementStorage.TradingManagementStorage
            storage tms = LibTradingManagementStorage
                .tradingManagementStorage();
        return tms.dataAccessExpirationSubscription[_buyer][tokenId];
    }

    // Returns the expiration time in unixtime if subscription exists
    // It will return 0, if subscription does not exist.
    // Only the most recent subscription time is reported (since only the most
    // recent access purchase time is kept)
    function getLastDataAccessSubscriptionExpirationTime(
        address _buyer,
        uint256 tokenId
    ) external view returns (uint256) {
        LibTradingManagementStorage.TradingManagementStorage
            storage tms = LibTradingManagementStorage
                .tradingManagementStorage();

        if (tms.dataAccessExpirationSubscription[_buyer][tokenId].length > 0) {
            uint lastOne = tms
            .dataAccessExpirationSubscription[_buyer][tokenId].length - 1;
            return
                tms.dataAccessExpirationSubscription[_buyer][tokenId][lastOne];
        } else {
            return 0;
        }
    }

    // Returns the expiration time in unixtime if PAYU exists
    // It will return 0, if PAYU does not exist.
    // Only the most recent PAYU time is reported (since only the most
    // recent access purchase time is kept)
    function getLastDataAccessPAYUExpirationTime(
        address _buyer,
        uint256 tokenId
    ) external view returns (uint256) {
        LibTradingManagementStorage.TradingManagementStorage
            storage tms = LibTradingManagementStorage
                .tradingManagementStorage();

        if (tms.dataAccessExpirationPAYU[_buyer][tokenId].length > 0) {
            uint lastOne = tms
            .dataAccessExpirationPAYU[_buyer][tokenId].length - 1;
            return tms.dataAccessExpirationPAYU[_buyer][tokenId][lastOne];
        } else {
            return 0;
        }
    }

    // Return all offerIds as an array for a specific user
    function getAllUserDataAccessSubscriptionOfferIds(
        address user
    ) external view returns (uint256[] memory) {
        LibTradingManagementStorage.TradingManagementStorage
            storage tms = LibTradingManagementStorage
                .tradingManagementStorage();

        uint256 length = tms.userOfferIdsSUB[user].length();
        uint256[] memory offerIdList = new uint256[](length);

        for (uint256 i = 0; i < length; i++) {
            offerIdList[i] = tms.userOfferIdsSUB[user].at(i);
        }

        return offerIdList;
    }

    // Get how many offerIds this user has
    function getUserDataAccessSubscriptionOfferIdCount(
        address user
    ) external view returns (uint256) {
        LibTradingManagementStorage.TradingManagementStorage
            storage tms = LibTradingManagementStorage
                .tradingManagementStorage();
        return tms.userOfferIdsSUB[user].length();
    }

    // Get offerId at specific index for this user
    function getIthUserDataAccessSubscriptionOfferId(
        address user,
        uint256 index
    ) external view returns (uint256) {
        LibTradingManagementStorage.TradingManagementStorage
            storage tms = LibTradingManagementStorage
                .tradingManagementStorage();
        require(
            index < tms.userOfferIdsSUB[user].length(),
            'Offer Id set index out of bounds'
        );
        return tms.userOfferIdsSUB[user].at(index);
    }

    // Remove user offer and its data
    function deleteUserDataAccessSubscriptionOfferId(
        address user,
        uint256 offerId
    ) external {
        LibDiamond.enforceIsContractOwner();
        LibTradingManagementStorage.TradingManagementStorage
            storage tms = LibTradingManagementStorage
                .tradingManagementStorage();
        delete tms.dataAccessExpirationSubscription[user][offerId];
        tms.userOfferIdsSUB[user].remove(offerId);
    }

    // Return all offerIds as an array for a specific user
    function getAllUserDataAccessPAYGOfferIds(
        address user
    ) external view returns (uint256[] memory) {
        LibTradingManagementStorage.TradingManagementStorage
            storage tms = LibTradingManagementStorage
                .tradingManagementStorage();

        uint256 length = tms.userOfferIdsPAYG[user].length();
        uint256[] memory offerIdList = new uint256[](length);

        for (uint256 i = 0; i < length; i++) {
            offerIdList[i] = tms.userOfferIdsPAYG[user].at(i);
        }

        return offerIdList;
    }

    // Get how many offerIds this user has
    function getUserDataAccessPAYGOfferIdCount(
        address user
    ) external view returns (uint256) {
        LibTradingManagementStorage.TradingManagementStorage
            storage tms = LibTradingManagementStorage
                .tradingManagementStorage();
        return tms.userOfferIdsPAYG[user].length();
    }

    // Get offerId at specific index for this user
    function getIthUserDataAccessPAYGOfferId(
        address user,
        uint256 index
    ) external view returns (uint256) {
        LibTradingManagementStorage.TradingManagementStorage
            storage tms = LibTradingManagementStorage
                .tradingManagementStorage();
        require(
            index < tms.userOfferIdsPAYG[user].length(),
            'Offer Id set index out of bounds'
        );
        return tms.userOfferIdsPAYG[user].at(index);
    }

    // Remove user offer and its data
    function deleteUserDataAccessPAYGOfferId(
        address user,
        uint256 offerId
    ) external {
        LibDiamond.enforceIsContractOwner();
        LibTradingManagementStorage.TradingManagementStorage
            storage tms = LibTradingManagementStorage
                .tradingManagementStorage();
        delete tms.dataAccessExpirationPAYG[user][offerId];
        tms.userOfferIdsPAYG[user].remove(offerId);
    }

    // Return all offerIds as an array for a specific user
    function getAllUserDataAccessPAYUOfferIds(
        address user
    ) external view returns (uint256[] memory) {
        LibTradingManagementStorage.TradingManagementStorage
            storage tms = LibTradingManagementStorage
                .tradingManagementStorage();

        uint256 length = tms.userOfferIdsPAYU[user].length();
        uint256[] memory offerIdList = new uint256[](length);

        for (uint256 i = 0; i < length; i++) {
            offerIdList[i] = tms.userOfferIdsPAYU[user].at(i);
        }

        return offerIdList;
    }

    // Get how many offerIds this user has
    function getUserDataAccessPAYUOfferIdCount(
        address user
    ) external view returns (uint256) {
        LibTradingManagementStorage.TradingManagementStorage
            storage tms = LibTradingManagementStorage
                .tradingManagementStorage();
        return tms.userOfferIdsPAYU[user].length();
    }

    // Get offerId at specific index for this user
    function getIthUserDataAccessPAYUOfferId(
        address user,
        uint256 index
    ) external view returns (uint256) {
        LibTradingManagementStorage.TradingManagementStorage
            storage tms = LibTradingManagementStorage
                .tradingManagementStorage();
        require(
            index < tms.userOfferIdsPAYU[user].length(),
            'Offer Id set index out of bounds'
        );
        return tms.userOfferIdsPAYU[user].at(index);
    }

    // Remove user offer and its data
    function deleteUserDataAccessPAYUOfferId(
        address user,
        uint256 offerId
    ) external {
        LibDiamond.enforceIsContractOwner();
        LibTradingManagementStorage.TradingManagementStorage
            storage tms = LibTradingManagementStorage
                .tradingManagementStorage();
        delete tms.dataAccessExpirationPAYU[user][offerId];
        tms.userOfferIdsPAYU[user].remove(offerId);
    }

    // Burn access token
    function burnAccessToken(
        address _from,
        string memory _oid,
        bytes memory _data
    ) external {
        LibDiamond.enforceIsContractOwner();
        LibTradingManagementStorage.TradingManagementStorage
            storage tms = LibTradingManagementStorage
                .tradingManagementStorage();

        // Get offering token contract
        address offeringTokenContract = tms.offeringTokenContract;
        OfferingToken offeringToken = OfferingToken(offeringTokenContract);

        // Get asset ID from OID
        uint256 _assetId = offeringToken.getIDHash(_oid);

        // TODO: Burn token by calling the appropriate Data Access contract

        // Emit event
        emit DataAccessBurned(_from, _assetId, _data);
    }

    // Implement IERC1155Receiver functions
    function onERC1155Received(
        address,
        address,
        uint256,
        uint256,
        bytes memory
    ) public virtual returns (bytes4) {
        return this.onERC1155Received.selector;
    }

    function onERC1155BatchReceived(
        address,
        address,
        uint256[] memory,
        uint256[] memory,
        bytes memory
    ) public virtual returns (bytes4) {
        return this.onERC1155BatchReceived.selector;
    }

    // // ERC165 support
    // function supportsInterface(bytes4 interfaceId) public view virtual returns (bool) {
    //     return
    //         interfaceId == type(IERC1155Receiver).interfaceId ||
    //         interfaceId == type(IERC165).interfaceId;
    // }
}
