// SPDX-License-Identifier: MIT
pragma solidity ^0.8.0;

// Import LibDiamond for access control
import './diamond/libraries/LibDiamond.sol';
import './TradingManagementStorageLib.sol';

contract TradingManagementStorageFacet {
    // Constants for contract names (ensure they are 32 bytes or less)
    bytes32 constant BOURSE_CONTRACT = 'BourseContract';
    bytes32 constant OFFERING_TOKEN_CONTRACT = 'OfferingTokenContract';
    bytes32 constant PAYMENT_TOKEN_CONTRACT = 'PaymentTokenContract';
    bytes32 constant PAYG_DATA_ACCESS_CONTRACT = 'PaygDataAccessContract';
    bytes32 constant PAYU_DATA_ACCESS_CONTRACT = 'PayuDataAccessContract';
    bytes32 constant SUBSCRIPTION_DATA_ACCESS_CONTRACT =
        'SubscriptionDataAccessContract';

    // Event to log contract address updates
    event ContractAddressUpdated(
        bytes32 indexed contractName,
        address previousAddress,
        address indexed newAddress
    );

    // Setter functions with event emission
    function setBourseContractAddress(address _address) external {
        LibDiamond.enforceIsContractOwner();
        LibTradingManagementStorage.TradingManagementStorage
            storage tms = LibTradingManagementStorage
                .tradingManagementStorage();
        address previousAddress = tms.bourseContractAddress;
        tms.bourseContractAddress = _address;
        emit ContractAddressUpdated(BOURSE_CONTRACT, previousAddress, _address);
    }

    function setOfferingTokenContract(address _address) external {
        LibDiamond.enforceIsContractOwner();
        LibTradingManagementStorage.TradingManagementStorage
            storage tms = LibTradingManagementStorage
                .tradingManagementStorage();
        address previousAddress = tms.offeringTokenContract;
        tms.offeringTokenContract = _address;
        emit ContractAddressUpdated(
            OFFERING_TOKEN_CONTRACT,
            previousAddress,
            _address
        );
    }

    function setPaymentTokenContract(address _address) external {
        LibDiamond.enforceIsContractOwner();
        LibTradingManagementStorage.TradingManagementStorage
            storage tms = LibTradingManagementStorage
                .tradingManagementStorage();
        address previousAddress = tms.paymentTokenContract;
        tms.paymentTokenContract = _address;
        emit ContractAddressUpdated(
            PAYMENT_TOKEN_CONTRACT,
            previousAddress,
            _address
        );
    }

    function setPaygDataAccessContract(address _address) external {
        LibDiamond.enforceIsContractOwner();
        LibTradingManagementStorage.TradingManagementStorage
            storage tms = LibTradingManagementStorage
                .tradingManagementStorage();
        address previousAddress = tms.paygDataAccessContract;
        tms.paygDataAccessContract = _address;
        emit ContractAddressUpdated(
            PAYG_DATA_ACCESS_CONTRACT,
            previousAddress,
            _address
        );
    }

    function setPayuDataAccessContract(address _address) external {
        LibDiamond.enforceIsContractOwner();
        LibTradingManagementStorage.TradingManagementStorage
            storage tms = LibTradingManagementStorage
                .tradingManagementStorage();
        address previousAddress = tms.payuDataAccessContract;
        tms.payuDataAccessContract = _address;
        emit ContractAddressUpdated(
            PAYU_DATA_ACCESS_CONTRACT,
            previousAddress,
            _address
        );
    }

    function setSubscriptionDataAccessContract(address _address) external {
        LibDiamond.enforceIsContractOwner();
        LibTradingManagementStorage.TradingManagementStorage
            storage tms = LibTradingManagementStorage
                .tradingManagementStorage();
        address previousAddress = tms.subscriptionDataAccessContract;
        tms.subscriptionDataAccessContract = _address;
        emit ContractAddressUpdated(
            SUBSCRIPTION_DATA_ACCESS_CONTRACT,
            previousAddress,
            _address
        );
    }

    // Function to set all contract addresses in one call
    function setContracts(
        address _bourseContractAddress,
        address _offeringTokenContract,
        address _paymentTokenContract,
        address _paygDataAccessContract,
        address _payuDataAccessContract,
        address _subscriptionDataAccessContract
    ) external {
        LibDiamond.enforceIsContractOwner();
        LibTradingManagementStorage.TradingManagementStorage
            storage tms = LibTradingManagementStorage
                .tradingManagementStorage();

        // Update and emit events for each contract address if changed
        if (tms.bourseContractAddress != _bourseContractAddress) {
            address previousAddress = tms.bourseContractAddress;
            tms.bourseContractAddress = _bourseContractAddress;
            emit ContractAddressUpdated(
                BOURSE_CONTRACT,
                previousAddress,
                _bourseContractAddress
            );
        }

        if (tms.offeringTokenContract != _offeringTokenContract) {
            address previousAddress = tms.offeringTokenContract;
            tms.offeringTokenContract = _offeringTokenContract;
            emit ContractAddressUpdated(
                OFFERING_TOKEN_CONTRACT,
                previousAddress,
                _offeringTokenContract
            );
        }

        if (tms.paymentTokenContract != _paymentTokenContract) {
            address previousAddress = tms.paymentTokenContract;
            tms.paymentTokenContract = _paymentTokenContract;
            emit ContractAddressUpdated(
                PAYMENT_TOKEN_CONTRACT,
                previousAddress,
                _paymentTokenContract
            );
        }

        if (tms.paygDataAccessContract != _paygDataAccessContract) {
            address previousAddress = tms.paygDataAccessContract;
            tms.paygDataAccessContract = _paygDataAccessContract;
            emit ContractAddressUpdated(
                PAYG_DATA_ACCESS_CONTRACT,
                previousAddress,
                _paygDataAccessContract
            );
        }

        if (tms.payuDataAccessContract != _payuDataAccessContract) {
            address previousAddress = tms.payuDataAccessContract;
            tms.payuDataAccessContract = _payuDataAccessContract;
            emit ContractAddressUpdated(
                PAYU_DATA_ACCESS_CONTRACT,
                previousAddress,
                _payuDataAccessContract
            );
        }

        if (
            tms.subscriptionDataAccessContract !=
            _subscriptionDataAccessContract
        ) {
            address previousAddress = tms.subscriptionDataAccessContract;
            tms
                .subscriptionDataAccessContract = _subscriptionDataAccessContract;
            emit ContractAddressUpdated(
                SUBSCRIPTION_DATA_ACCESS_CONTRACT,
                previousAddress,
                _subscriptionDataAccessContract
            );
        }
    }

    // Getter functions
    function getBourseContractAddress() external view returns (address) {
        return
            LibTradingManagementStorage
                .tradingManagementStorage()
                .bourseContractAddress;
    }

    function getOfferingTokenContract() external view returns (address) {
        return
            LibTradingManagementStorage
                .tradingManagementStorage()
                .offeringTokenContract;
    }

    function getPaymentTokenContract() external view returns (address) {
        return
            LibTradingManagementStorage
                .tradingManagementStorage()
                .paymentTokenContract;
    }

    function getPaygDataAccessContract() external view returns (address) {
        return
            LibTradingManagementStorage
                .tradingManagementStorage()
                .paygDataAccessContract;
    }

    function getPayuDataAccessContract() external view returns (address) {
        return
            LibTradingManagementStorage
                .tradingManagementStorage()
                .payuDataAccessContract;
    }

    function getSubscriptionDataAccessContract()
        external
        view
        returns (address)
    {
        return
            LibTradingManagementStorage
                .tradingManagementStorage()
                .subscriptionDataAccessContract;
    }
}
