import { ethers } from 'hardhat';
import { FameContracts } from './deploy-helper';
import { Signers } from '../types';
import {
  DiamondLoupeFacet,
  TradingManagementStorageFacet,
  TradingManagementExecutorFacet,
} from '../../../../types';

import { initializeTradingManagmentStorageContract as init } from '../../../scripts/initializeTradingManagementStorage';

export async function initializeContracts({
  contracts,
  signers,
}: {
  contracts: FameContracts;
  signers: Signers;
}): Promise<void> {
  // Deploy the contracts as before
  const {
    paymentTokenContract,
    bourseContract,
    escrowContract,
    governanceContract,
  } = contracts;

  // Use the admin signer to initialize the contracts
  const adminSigner = signers.admin;

  // Connect contracts with the admin signer
  const paymentToken = paymentTokenContract.connect(adminSigner);
  const bourse = bourseContract.connect(adminSigner);
  const escrow = escrowContract.connect(adminSigner);
  const governance = governanceContract.connect(adminSigner);
  await (await governance.initialize()).wait();
  await (await bourse.initialize(escrowContract.address)).wait();

  // Transfer ownership of PaymentToken to Governance contract if not already done
  const ownerPayment = await paymentToken.owner();
  if (ownerPayment !== governance.address) {
    await (await paymentToken.transferOwnership(governance.address)).wait();
  }

  // Transfer ownership of Bourse to Governance contract if not already done
  const ownerBourse = await bourse.owner();
  if (ownerBourse !== governance.address) {
    await (await bourse.transferOwnership(governance.address)).wait();
  }

  // Set Bourse address in Escrow contract if not already set
  const bourseAddress = await escrow.getBourse();
  if (bourseAddress === ethers.constants.AddressZero) {
    await (await escrow.setBourse(bourse.address)).wait();
  }

  // Set up Bourse Contract in Governance contract
  await (await governance.setFameBourse(bourse.address)).wait();
  const tx = await governance.registerCoinToken(paymentToken.address, 0, 'FDE');
  await tx.wait();
}

export async function initializeTradingManagmentContract({
  signers,
  contracts,
  tradingManagementAddress,
}: {
  contracts: FameContracts;
  signers: Signers;
  tradingManagementAddress: string;
}): Promise<{
  diamondLoupeFacet: DiamondLoupeFacet;
  tradingManagementStorageFacet: TradingManagementStorageFacet;
  tradingManagementExecutorFacet: TradingManagementExecutorFacet;
}> {
  // Get instances of the core facets using the Diamond's address

  const contractAddresses = {
    contractAddressPaymentToken: contracts.paymentTokenContract.address,
    contractAddressOfferingToken: contracts.offeringTokenContract.address,
    contractAddressBourse: contracts.bourseContract.address,
    contractAddressDataAccessPAYG:
      contracts.dataAccessPayAsYouGoContract.address,
    contractAddressDataAccessPAYU:
      contracts.dataAccessPayAsYouUseContract.address,
    contractAddressDataAccessSubscription:
      contracts.dataAccessSubscriptionContract.address,
    contractAddressGovernance: contracts.governanceContract.address,
    contractAddressEscrow: contracts.escrowContract.address,
    contractAddressTradingManagement: tradingManagementAddress,
  };

  return init({ contracts: contractAddresses });
}
