import { ethers } from 'hardhat';
import { DiamondCutFacet } from '../../types';
import { getSelectors, FacetCutAction } from './libraries/diamond';
import {
  DiamondFullInfo,
  FacetInfo,
  saveContractAddresses,
  saveDiamondInfo,
} from '../contract-helpers';
import { getSigners } from '../contracts/test/helpers/deploy-helper';

export let DiamondAddress: string;

// Deploying BourseFacet Libs
export async function deployBourseLibs(): Promise<any> {
  const avlTreeLib = await ethers.getContractFactory('AVLTreeLib');
  const tokenUtilsLib = await ethers.getContractFactory('TokenUtilsLib');

  const avlTreeLibDeployment = await avlTreeLib.deploy();
  await avlTreeLibDeployment.deployed();
  const tokenUtilsLibDeployment = await tokenUtilsLib.deploy();
  await tokenUtilsLibDeployment.deployed();

  const bourseLib = await ethers.getContractFactory('BourseLib', {
    libraries: {
      AVLTreeLib: avlTreeLibDeployment.address,
      TokenUtilsLib: tokenUtilsLibDeployment.address,
    },
  });
  const bourseLibDeployment = await bourseLib.deploy();
  await bourseLibDeployment.deployed();
  return {
    avlTreeLibAddress: avlTreeLibDeployment.address,
    tokenUtilsLibAddress: tokenUtilsLibDeployment.address,
    bourseLibAddress: bourseLibDeployment.address,
  };
}

export async function deployDiamond(
  props: { saveInfo: boolean; showInfo: boolean } = {
    saveInfo: true,
    showInfo: true,
  },
): Promise<string> {
  if (!props.showInfo) {
    console.log = () => {
      return;
    };
  }

  console.log(`
-----------------------------------------------------
- Deploying Diamond Bourse contract...
- Network: ${process.env.HARDHAT_NETWORK}
-----------------------------------------------------
    `);

  const signers = await getSigners();
  const contractOwner = signers.admin;

  // deploy DiamondCutFacet
  const DiamondCutFacet = await ethers.getContractFactory('DiamondCutFacet');
  const diamondCutFacet = await DiamondCutFacet.deploy();
  await diamondCutFacet.deployed();
  console.log('DiamondCutFacet deployed:', diamondCutFacet.address);

  // deploy Diamond
  const Diamond = await ethers.getContractFactory('Bourse');
  const diamond = await Diamond.deploy(
    contractOwner.address,
    diamondCutFacet.address,
  );
  await diamond.deployed();
  console.log('Diamond deployed:', diamond.address);

  // deploy DiamondInit
  // DiamondInit provides a function that is called when the diamond is upgraded to initialize state variables
  // Read about how the diamondCut function works here: https://eips.ethereum.org/EIPS/eip-2535#addingreplacingremoving-functions
  const DiamondInit = await ethers.getContractFactory('DiamondInit');
  const diamondInit = await DiamondInit.deploy();
  await diamondInit.deployed();
  console.log('DiamondInit deployed:', diamondInit.address);
  // deploy facets
  console.log('');
  console.log('Deploying default facets');
  const FacetNames = ['DiamondLoupeFacet', 'OwnershipFacet', 'BourseFacet'];

  const facetsInfo: FacetInfo[] = [];

  facetsInfo.push({
    publishData: new Date().toISOString(),
    facetName: 'DiamondCutFacet',
    facetAddress: diamondCutFacet.address,
  });

  /******* the following is replaced with the function deployBourseLibs() 
  // Deploying BourseFacet Libs
  const avlTreeLib = await ethers.getContractFactory('AVLTreeLib');
  const tokenUtilsLib = await ethers.getContractFactory('TokenUtilsLib');

  const avlTreeLibDeployment = await avlTreeLib.deploy();
  await avlTreeLibDeployment.deployed();
  const tokenUtilsLibDeployment = await tokenUtilsLib.deploy();
  await tokenUtilsLibDeployment.deployed();

  const bourseLib = await ethers.getContractFactory('BourseLib', {
    libraries: {
      AVLTreeLib: avlTreeLibDeployment.address,
      TokenUtilsLib: tokenUtilsLibDeployment.address,
    },
  });
  const bourseLibDeployment = await bourseLib.deploy();
  await bourseLibDeployment.deployed();
  *******/
  const bourseLibAddresses = await deployBourseLibs();

  const diamondCutList = [];
  for (const FacetName of FacetNames) {
    let Facet;
    if (FacetName === 'BourseFacet') {
      Facet = await ethers.getContractFactory(FacetName, {
        libraries: {
          AVLTreeLib: bourseLibAddresses.avlTreeLibAddress,
          TokenUtilsLib: bourseLibAddresses.tokenUtilsLibAddress,
          BourseLib: bourseLibAddresses.bourseLibAddress,
        },
      });
    } else {
      Facet = await ethers.getContractFactory(FacetName);
    }
    const facet = await Facet.deploy();
    await facet.deployed();
    console.log(`${FacetName} deployed: ${facet.address}`);

    let selectors = getSelectors(facet);
    // console.log(`Selectors for ${FacetName}:`, selectors);

    // selector for 'supportsInterface(bytes4)' is multiply defined, therefore remove it.
    selectors = selectors.filter((str) => str !== '0x01ffc9a7');

    diamondCutList.push({
      facetAddress: facet.address,
      action: FacetCutAction.Add,
      functionSelectors: selectors,
    });
    facetsInfo.push({
      publishData: new Date().toISOString(),
      facetName: FacetName,
      facetAddress: facet.address,
    });
  }

  console.log(
    '---------------  Upgrading diamond with facets -------------------------',
  );
  console.log('Initiating diamond cut');
  // console.log('Diamond Cut List:', diamondCutList);
  const diamondCutDeployedFacet = (await ethers.getContractAt(
    'IDiamondCut',
    diamond.address,
  )) as DiamondCutFacet;

  // call to init function
  const functionCall = diamondInit.interface.encodeFunctionData('init');
  const tx = await diamondCutDeployedFacet.diamondCut(
    diamondCutList,
    diamondInit.address,
    functionCall,
  );
  console.log('Diamond cut tx: ', tx.hash);
  const receipt = await tx.wait();
  if (!receipt.status) {
    throw Error(`Diamond upgrade failed: ${tx.hash}`);
  }
  console.log(`Completed diamond cut by upgrading facets`);
  DiamondAddress = diamond.address;

  const diamondFullInfo: DiamondFullInfo = {
    name: 'Bourse',
    diamondAddress: DiamondAddress,
    diamondInitAddress: diamondInit.address,
    ownership: contractOwner.address,
    publishData: new Date().toISOString(),
    facets: facetsInfo.reduce(
      (acc, facet) => {
        acc[facet.facetName] = [facet];
        return acc;
      },
      {} as Record<string, FacetInfo[]>,
    ),
  };

  if (props.saveInfo) {
    saveDiamondInfo(process.env.HARDHAT_NETWORK, diamondFullInfo);
    saveContractAddresses(process.env.HARDHAT_NETWORK, {
      [`contractAddress${diamondFullInfo.name}`]: DiamondAddress,
    });
  } else {
    console.log('Diamond info:', diamondFullInfo);
  }
  return Promise.resolve(DiamondAddress);
}

// We recommend this pattern to be able to use async/await everywhere
// and properly handle errors.
if (require.main === module) {
  deployDiamond()
    .then(() => process.exit(0))
    .catch((error) => {
      console.error(error);
      process.exit(1);
    });
}

exports.deployDiamond = deployDiamond;
exports.deployBourseLibs = deployBourseLibs;
