import { ethers, upgrades } from 'hardhat';
import '@openzeppelin/hardhat-upgrades';
import * as dotenv from 'dotenv';
import { saveContractAddresses } from '../contract-helpers';
import {
  Bourse,
  DataAccessPayAsYouGo,
  DataAccessPayAsYouUse,
  DataAccessSubscription,
  Escrow,
  Governance,
  OfferingToken,
  PaymentToken,
} from '../../types';
const envFile = process.env.NODE_ENV ? `.env.${process.env.NODE_ENV}` : '.env';

const result = dotenv.config({ path: envFile });

if (result.error) {
  console.warn(`Env file not found: ${envFile}, skipping dotenv config`);
} else {
  console.log(`Successfully loaded env file: ${envFile}`);
}

export interface FameContracts {
  paymentTokenContract: PaymentToken;
  escrowContract: Escrow;
  bourseContract: Bourse;
  offeringTokenContract: OfferingToken;
  dataAccessPayAsYouGoContract: DataAccessPayAsYouGo;
  dataAccessPayAsYouUseContract: DataAccessPayAsYouUse;
  dataAccessSubscriptionContract: DataAccessSubscription;
  governanceContract: Governance;
}

export async function deployContracts(props?: {
  shouldSaveInfo: boolean;
}): Promise<FameContracts> {
  const [deployer] = await ethers.getSigners();

  if (props && !props.shouldSaveInfo) {
    console.log = () => {
      void 0;
    };
  }

  console.log(
    `Deploying contracts on network ${process.env.HARDHAT_NETWORK} using account ${deployer.address}`,
  );

  // Deploy PaymentToken contract
  const PaymentTokenContractFactory =
    await ethers.getContractFactory('PaymentToken');
  const paymentToken = (await upgrades.deployProxy(
    PaymentTokenContractFactory,
    ['PaymentToken', 'FDE'],
  )) as PaymentToken;
  await paymentToken.deployed();
  const paymentContractAddress = paymentToken.address;
  console.log('PaymentToken deployed to:', paymentContractAddress);

  // Deploy Escrow contract
  const escrowContract = await ethers.getContractFactory('Escrow');
  const escrow = (await escrowContract.deploy(
    paymentContractAddress,
  )) as Escrow;
  await escrow.deployed();
  const escrowContractAddress = escrow.address;
  console.log('Escrow deployed to:', escrow.address);

  /**********************  the following are replaced by upgradable Diamond contract
  // Deploying Libs and Bourse contract
  const avlTreeLib = await ethers.getContractFactory('AVLTreeLib');
  const tokenUtilsLib = await ethers.getContractFactory('TokenUtilsLib');

  const avlTreeLibDeployment = await avlTreeLib.deploy();
  await avlTreeLibDeployment.deployed();
  const tokenUtilsLibDeployment = await tokenUtilsLib.deploy();
  await tokenUtilsLibDeployment.deployed();

  const bourseLib = await ethers.getContractFactory('BourseLib', {
    libraries: {
      AVLTreeLib: avlTreeLibDeployment.address,
      TokenUtilsLib: tokenUtilsLibDeployment.address,
    },
  });
  const bourseLibDeployment = await bourseLib.deploy();
  await bourseLibDeployment.deployed();

  // Deploy Bourse contract
  const bourse = await ethers.getContractFactory('Bourse', {
    libraries: {
      AVLTreeLib: avlTreeLibDeployment.address,
      TokenUtilsLib: tokenUtilsLibDeployment.address,
      BourseLib: bourseLibDeployment.address,
    },
  });

  const bourseContract = (await bourse.deploy(escrowContractAddress)) as Bourse;
  await bourseContract.deployed();
  const bourseContractAddress = bourseContract.address;
  console.log('Bourse contract deployed to:', bourseContractAddress);
  *********************************************/

  // Deploy OfferingToken contract
  const OfferingTokenContractFactory =
    await ethers.getContractFactory('OfferingToken');
  const offeringToken = (await upgrades.deployProxy(
    OfferingTokenContractFactory,
    ['OfferingToken', 'OFRNG'],
  )) as OfferingToken;
  await offeringToken.deployed();
  const offeringContractAddress = offeringToken.address;
  console.log('OfferingToken deployed to:', offeringContractAddress);

  // Deploy DataAccessPayAsYouGo contract
  const DataAccessPayAsYouGoContractFactory = await ethers.getContractFactory(
    'DataAccessPayAsYouGo',
  );
  const dataAccessPayAsYouGo = (await upgrades.deployProxy(
    DataAccessPayAsYouGoContractFactory,
    ['PayAsYouGo', 'PAYG', ''],
  )) as DataAccessPayAsYouGo;
  dataAccessPayAsYouGo.deployed();
  console.log(
    'DataAccessPayAsYouGo deployed to:',
    dataAccessPayAsYouGo.address,
  );

  // Deploy DataAccessPayAsYouUse contract
  const DataAccessPayAsYouUseContractFactory = await ethers.getContractFactory(
    'DataAccessPayAsYouUse',
  );
  const dataAccessPayAsYouUse = (await upgrades.deployProxy(
    DataAccessPayAsYouUseContractFactory,
    ['PayAsYouUse', 'PAYU', ''],
  )) as DataAccessPayAsYouUse;
  await dataAccessPayAsYouUse.deployed();
  console.log(
    'DataAccessPayAsYouUse deployed to:',
    dataAccessPayAsYouUse.address,
  );

  // Deploy DataAccessSubscription contract
  const DataAccessSubscriptionContractFactory = await ethers.getContractFactory(
    'DataAccessSubscription',
  );
  const dataAccessSubscription = (await upgrades.deployProxy(
    DataAccessSubscriptionContractFactory,
    ['Subscription', 'SUBS', ''],
  )) as DataAccessSubscription;
  await dataAccessSubscription.deployed();
  console.log(
    'DataAccessSubscription deployed to:',
    dataAccessSubscription.address,
  );

  /**********************  the following are replaced by upgradable Diamond contract
  // Deploy Governance contract
  const governanceContractContract =
    await ethers.getContractFactory('Governance');
  const governanceContract =
    (await governanceContractContract.deploy()) as Governance;
  await governanceContract.deployed();
  const governanceContractAddress = governanceContract.address;
  console.log('Governance contract deployed to:', governanceContractAddress);
  *********************************************/

  const contractAddresses = {
    contractAddressPaymentToken: paymentContractAddress,
    contractAddressOfferingToken: offeringContractAddress,
    // contractAddressBourse: bourseContractAddress,
    contractAddressDataAccessPAYG: dataAccessPayAsYouGo.address,
    contractAddressDataAccessPAYU: dataAccessPayAsYouUse.address,
    contractAddressDataAccessSubscription: dataAccessSubscription.address,
    // contractAddressGovernance: governanceContractAddress,
    contractAddressEscrow: escrowContractAddress,
  };

  if (!props || props.shouldSaveInfo) {
    saveContractAddresses(process.env.HARDHAT_NETWORK, contractAddresses);
  }
  return {
    paymentTokenContract: paymentToken,
    escrowContract: escrow,
    // bourseContract: bourseContract,
    offeringTokenContract: offeringToken,
    dataAccessPayAsYouGoContract: dataAccessPayAsYouGo,
    dataAccessPayAsYouUseContract: dataAccessPayAsYouUse,
    dataAccessSubscriptionContract: dataAccessSubscription,
    // governanceContract: governanceContract,
  };
}

if (require.main === module) {
  deployContracts()
    .then(() => process.exit(0))
    .catch((error) => {
      console.error(error);
      process.exit(1);
    });
}
