// scripts/initializeContracts.ts

import { ethers } from 'hardhat';
import {
  PaymentToken__factory,
  BourseFacet__factory,
  OwnershipFacet__factory,
  GovernanceFacet__factory,
  Escrow__factory,
} from '../../types';
import * as dotenv from 'dotenv';
import { getContractAddresses } from '../contract-helpers';
import { getSigners } from '../contracts/test/helpers/deploy-helper';

const envFile = process.env.NODE_ENV ? `.env.${process.env.NODE_ENV}` : '.env';
console.log(`Loading env file: ${envFile}`);
dotenv.config({ path: envFile });

export async function initializeGovAndBourseContracts() {
  const provider = ethers.provider;

  const signers = await getSigners();

  const contractAddresses = getContractAddresses(process.env.HARDHAT_NETWORK);

  const contractPaymentToken = PaymentToken__factory.connect(
    contractAddresses.contractAddressPaymentToken,
    provider,
  );
  const contractGovernance = GovernanceFacet__factory.connect(
    contractAddresses.contractAddressGovernance,
    provider,
  );
  const contractEscrow = Escrow__factory.connect(
    contractAddresses.contractAddressEscrow,
    provider,
  );

  console.log(
    `Starting initialization of contracts using admin signer ${signers.admin.address}`,
  );

  const contractBourse1 = BourseFacet__factory.connect(
    contractAddresses.contractAddressBourse,
    provider,
  );

  await initializeGovernance(contractGovernance, signers.admin);
  await initializeBourse(
    contractBourse1,
    contractAddresses.contractAddressEscrow,
    signers.admin,
  );

  const contractBourse = OwnershipFacet__factory.connect(
    contractAddresses.contractAddressBourse,
    provider,
  );

  await transferPaymentContractOwnership(
    contractPaymentToken,
    contractGovernance,
    signers.admin,
  );
  await transferBourseContractOwnership(
    contractBourse,
    contractGovernance,
    signers.admin,
  );
  await transferEscrowContractOwnership(
    contractEscrow,
    contractBourse,
    signers.admin,
  );
  await setFameBourseInGovernance(
    contractGovernance,
    contractBourse,
    signers.admin,
  );
  await registeringFDETokenInGovernance(
    contractGovernance,
    contractPaymentToken,
    signers.admin,
  );

  console.log(
    `Initialization of contracts using admin signer ${signers.admin.address} was successfully completed!`,
  );
}

async function transferPaymentContractOwnership(
  contractPaymentToken,
  contractGovernance,
  adminSigner,
) {
  try {
    const ownerPayment = await contractPaymentToken.owner();
    if (
      ownerPayment.toLowerCase() !== contractGovernance.address.toLowerCase()
    ) {
      const tx = await contractPaymentToken
        .connect(adminSigner)
        .transferOwnership(contractGovernance.address);
      await tx.wait(1);
      console.log(
        `Payment contract ${contractPaymentToken.address} ownership was successfully transferred to Governance Contract ${contractGovernance.address}!`,
      );
      const newOwnerPayment = await contractPaymentToken.owner();
      console.log(`Current payment contract owner is: ${newOwnerPayment}`);
    } else {
      console.log(
        `Governance contract is already the owner of Payment Token: ${ownerPayment}. Now need to set it up`,
      );
    }
  } catch (e) {
    console.error(
      `Error while transferring payment contract ownership to governance contract`,
      e,
    );
  }
}

async function transferBourseContractOwnership(
  contractBourse,
  contractGovernance,
  adminSigner,
) {
  try {
    const ownerBourse = await contractBourse.owner();
    if (
      ownerBourse.toLowerCase() !== contractGovernance.address.toLowerCase()
    ) {
      const tx = await contractBourse
        .connect(adminSigner)
        .transferOwnership(contractGovernance.address);
      await tx.wait(1);
      console.log(
        `Bourse contract ${contractBourse.address} ownership transferred to Governance Contract ${contractGovernance.address} successfully`,
      );
    } else {
      console.log(
        `Governance Contract ${contractGovernance.address} is already the owner of Bourse contract ${contractBourse.address}. Now need to set it up`,
      );
    }
  } catch (e) {
    console.error(
      `Failed! Bourse contract ${contractBourse.address} ownership was not transferred to Governance Contract ${contractGovernance.address}`,
      e,
    );
  }
}

async function transferEscrowContractOwnership(
  contractEscrow,
  contractBourse,
  adminSigner,
) {
  try {
    const bourseAddress = await contractEscrow.getBourse();
    if (bourseAddress === ethers.constants.AddressZero) {
      const tx = await contractEscrow
        .connect(adminSigner)
        .setBourse(contractBourse.address);
      await tx.wait(1);
      console.log(
        `Bourse contract ${contractBourse.address} set on Escrow contract ${contractEscrow.address} successfully!`,
      );
    } else {
      console.log(
        `Escrow ${contractEscrow.address} contract has already set up bourse contract ${contractBourse.address}`,
      );
    }
  } catch (e) {
    console.error(
      `Failed! Bourse contract ${contractBourse.address} was not set up on Escrow contract`,
      e,
    );
  }
}

async function setFameBourseInGovernance(
  contractGovernance,
  contractBourse,
  adminSigner,
) {
  try {
    const bourseAddress = await contractGovernance.getFameBourse();
    if (bourseAddress.toLowerCase() === contractBourse.address.toLowerCase()) {
      console.log(
        `Bourse contract ${bourseAddress} is already set up in Governance Contract`,
      );
    } else {
      console.log(
        `Setting up Bourse contract ${contractBourse.address} by setFameBourse method on Governance Contract ${contractGovernance.address}`,
      );
      const tx = await contractGovernance
        .connect(adminSigner)
        .setFameBourse(contractBourse.address);
      await tx.wait(1);
      console.log(
        `Fame Bourse Contract ${contractBourse.address} set in Governance contract ${contractGovernance.address} successfully!`,
      );
    }
  } catch (e) {
    console.error(`Failed to set Fame Bourse in Governance contract`, e);
  }
}

async function initializeBourse(
  contractBourse,
  contractAddressEscrow,
  adminSigner,
) {
  try {
    const tx = await contractBourse
      .connect(adminSigner)
      .initialize(contractAddressEscrow);
    await tx.wait(1);
    console.log(`Bourse initialized successfully!`);
  } catch (e) {
    console.error(`Failed to initialize Bourse`);
  }
}

async function initializeGovernance(contractGovernance, adminSigner) {
  try {
    const tx = await contractGovernance.connect(adminSigner).initialize();
    await tx.wait(1);
    console.log(`Governance initialized successfully!`);
  } catch (e) {
    console.error(`Failed to initialize Governance`);
  }
}

async function registeringFDETokenInGovernance(
  contractGovernance,
  contractPaymentToken,
  adminSigner,
) {
  try {
    const tx = await contractGovernance.getCoinAddress('FDE');
    if (
      tx[0].toLowerCase() === contractPaymentToken.address.toLowerCase() &&
      tx[1].eq(0)
    ) {
      console.log(
        `FDE Coin is already set up in Governance contract ${contractGovernance.address}`,
      );
    } else {
      console.error(
        `FDE Coin is not properly set up in Governance contract ${contractGovernance.address}`,
      );
      console.error(tx);
    }
  } catch (error) {
    if (error.message.includes('Coin does not exist')) {
      console.log(
        `Coin FDE is not visible via Governance Contract ${contractGovernance.address}`,
      );
      console.log('Starting registering FDE token in Governance Contract');
      const registerTx = await contractGovernance
        .connect(adminSigner)
        .registerCoinToken(contractPaymentToken.address, 0, 'FDE');
      await registerTx.wait(1);
      console.log(
        `FDE coin registered in the governance contract ${contractGovernance.address} successfully!`,
      );
    } else {
      console.error(
        `Failed to register FDE coin in Governance contract ${contractGovernance.address}`,
        error,
      );
    }
  }
}
