import { ContractAddresses, getContractAddresses } from '../contract-helpers';
import {
  DiamondLoupeFacet,
  TradingManagementStorageFacet,
  TradingManagementExecutorFacet,
} from '../../types/';
import { ethers } from 'hardhat';
import { getSigners } from '../contracts/test/helpers/deploy-helper';
import * as dotenv from 'dotenv';

const envFile = process.env.NODE_ENV ? `.env.${process.env.NODE_ENV}` : '.env';
console.log(`Loading env file: ${envFile}`);
dotenv.config({ path: envFile, override: true });

export async function initializeTradingManagmentStorageContract({
  contracts,
}: {
  contracts?: ContractAddresses;
}): Promise<{
  diamondLoupeFacet: DiamondLoupeFacet;
  tradingManagementStorageFacet: TradingManagementStorageFacet;
  tradingManagementExecutorFacet: TradingManagementExecutorFacet;
}> {
  console.log(
    `==> Initializing script to set up storage variables in the TradingManagementStorageFacet for the TradingManagement`,
  );
  if (!contracts) {
    contracts = getContractAddresses(process.env.HARDHAT_NETWORK);
    console.log(
      `==> Using contract addresses from the contract-helpers file for network ${
        process.env.HARDHAT_NETWORK
      }: ${JSON.stringify(contracts)}`,
    );
  }

  const signers = await getSigners();
  const owner = signers.admin;
  const diamondLoupeFacet = (await ethers.getContractAt(
    'DiamondLoupeFacet',
    contracts.contractAddressTradingManagement,
  )) as DiamondLoupeFacet;
  const tradingManagementStorageFacet = (await ethers.getContractAt(
    'TradingManagementStorageFacet',
    contracts.contractAddressTradingManagement,
  )) as TradingManagementStorageFacet;
  const tradingManagementExecutorFacet = (await ethers.getContractAt(
    'TradingManagementExecutorFacet',
    contracts.contractAddressTradingManagement,
  )) as TradingManagementExecutorFacet;

  console.log(
    `==> Setting storage variables in the TradingManagementStorageFacet for the TradingManagement contract at address: ${contracts.contractAddressTradingManagement}`,
  );

  // Set storage variables in the TradingManagementStorageFacet
  const tx1 = await tradingManagementStorageFacet
    .connect(owner)
    .setOfferingTokenContract(contracts.contractAddressOfferingToken);
  await tx1.wait(1);
  console.log(
    `==> Successfully set OfferingToken contract address in the TradingManagementStorageFacet`,
  );
  const tx2 = await tradingManagementStorageFacet
    .connect(owner)
    .setPaymentTokenContract(contracts.contractAddressPaymentToken);
  await tx2.wait(1);
  console.log(
    `==> Successfully set PaymentToken contract address in the TradingManagementStorageFacet`,
  );

  const tx3 = await tradingManagementStorageFacet
    .connect(owner)
    .setBourseContractAddress(contracts.contractAddressBourse);

  await tx3.wait(1);
  console.log(
    `==> Successfully set Bourse contract address in the TradingManagementStorageFacet`,
  );
  const tx4 = await tradingManagementStorageFacet
    .connect(owner)
    .setPaygDataAccessContract(contracts.contractAddressDataAccessPAYG);

  await tx4.wait(1);
  console.log(
    `==> Successfully set PaygDataAccess contract address in the TradingManagementStorageFacet`,
  );

  const tx5 = await tradingManagementStorageFacet
    .connect(owner)
    .setPayuDataAccessContract(contracts.contractAddressDataAccessPAYU);

  await tx5.wait(1);
  console.log(
    `==> Successfully set PayuDataAccess contract address in the TradingManagementStorageFacet`,
  );

  const tx6 = await tradingManagementStorageFacet
    .connect(owner)
    .setSubscriptionDataAccessContract(
      contracts.contractAddressDataAccessSubscription,
    );

  await tx6.wait(1);
  console.log(
    `==> Successfully set SubscriptionDataAccess contract address in the TradingManagementStorageFacet`,
  );

  console.log(
    `==> Successfully set all storage variables in the TradingManagementStorageFacet for the TradingManagement contract at address: ${contracts.contractAddressTradingManagement}`,
  );

  return {
    diamondLoupeFacet,
    tradingManagementStorageFacet,
    tradingManagementExecutorFacet,
  };
}
