/* eslint-disable @typescript-eslint/no-explicit-any */
/* eslint-disable @typescript-eslint/no-unused-vars */
import { ethers } from 'hardhat';
import { readFileSync } from 'fs';
import * as path from 'path';
import * as dotenv from 'dotenv';

// Import types for the facets
import { DiamondLoupeFacet } from '../../types';
import {
  DiamondFullInfo,
  readLastDeployedDiamondInfo,
} from '../contract-helpers';

const envFile = process.env.NODE_ENV ? `.env.${process.env.NODE_ENV}` : '.env';
console.log(`Loading env file: ${envFile}`);
dotenv.config({ path: envFile, override: true });

async function main() {
  // Read the Diamond address from your deployment script or configuration
  const diamondInfo: DiamondFullInfo = readLastDeployedDiamondInfo(
    process.env.HARDHAT_NETWORK,
    'TradingManagement',
  );

  const diamondAddress = diamondInfo.diamondAddress;
  console.log('Connected to Diamond at address:', diamondAddress);

  const accounts = await ethers.getSigners();
  const signer = accounts[0]; // Assuming the first account is the owner

  // Get the DiamondLoupeFacet interface connected to your Diamond
  const diamondLoupeFacet = (await ethers.getContractAt(
    'DiamondLoupeFacet',
    diamondAddress,
    signer,
  )) as DiamondLoupeFacet;

  // Retrieve all facets and their function selectors
  const facets = await diamondLoupeFacet.facets();

  console.log('Facets and their function selectors:');

  // Prepare a mapping from facet addresses to their names and ABIs
  const facetAddressToName: { [address: string]: string } = {};
  const facetABIs: { [address: string]: any } = {};

  // Read facets info from diamondInfo facets (get only last from the list of facets)
  const facetsInfo = Object.entries(diamondInfo.facets).map(
    ([key, value]) => value[value.length - 1],
  ); // Array of { facetName, facetAddress }

  // Build the mapping from facet addresses to facet names and load their ABIs
  for (const facetInfo of facetsInfo) {
    const facetAddress = facetInfo.facetAddress.toLowerCase();
    const facetName = facetInfo.facetName;

    facetAddressToName[facetAddress] = facetName;

    let facetFolder = facetName;

    switch (facetName) {
      case 'DiamondCutFacet':
      case 'DiamondInit':
      case 'DiamondLoupeFacet':
      case 'OwnershipFacet':
        facetFolder = `diamond/facets/${facetName}`;
        break;
    }

    // Load the ABI for this facet
    const abiPath = path.resolve(
      __dirname,
      `../../artifacts/libs/contracts/src/contracts/${facetFolder}.sol/${facetName}.json`,
    );
    const abiJson = JSON.parse(readFileSync(abiPath, 'utf8'));
    const facetInterface = new ethers.utils.Interface(abiJson.abi);

    facetABIs[facetAddress] = facetInterface;
  }

  // Iterate over each facet to display its functions
  for (const facet of facets) {
    const facetAddress = facet.facetAddress.toLowerCase();
    const functionSelectors = facet.functionSelectors;

    const facetName = facetAddressToName[facetAddress] || 'UnknownFacet';

    console.log(`\nFacet: ${facetName}`);
    console.log('Facet address:', facetAddress);
    console.log('Function selectors:', functionSelectors);

    const facetInterface = facetABIs[facetAddress];

    if (facetInterface) {
      console.log('Functions in this facet:');
      for (const selector of functionSelectors) {
        // Map the function selector back to the function signature
        const fragment = facetInterface.getFunction(selector);
        if (fragment) {
          console.log(`  ${selector}: ${fragment.format()}`);
        } else {
          console.log(`  ${selector}: Unknown function`);
        }
      }
    } else {
      console.log('No interface available for this facet.');
    }
  }

  // Check if specific functions are present in the Diamond
  const functionSignature = 'isAssetPurchaseble(string)';
  const functionSelector = ethers.utils
    .keccak256(ethers.utils.toUtf8Bytes(functionSignature))
    .substring(0, 10); // Get first 10 characters (4 bytes)

  console.log(
    `\nChecking for function ${functionSignature} with selector ${functionSelector}`,
  );

  const facetAddressForFunction =
    await diamondLoupeFacet.facetAddress(functionSelector);

  if (
    facetAddressForFunction !== '0x0000000000000000000000000000000000000000'
  ) {
    console.log(
      `Function ${functionSignature} is present in facet at address: ${facetAddressForFunction}`,
    );

    const facetName =
      facetAddressToName[facetAddressForFunction.toLowerCase()] ||
      'UnknownFacet';
    console.log(`Facet name: ${facetName}`);
  } else {
    console.log(`Function ${functionSignature} is not found in any facet`);
  }
}

main()
  .then(() => process.exit(0))
  .catch((error) => {
    console.error('Error:', error);
    process.exit(1);
  });
