import { ethers } from 'hardhat';
import { getDeployedDATContractAddress } from '../contract-helpers/contract-address-helpers';

export async function updateUri(
  tokenType: string,
  newUri: string,
  hre: any,
): Promise<void> {
  const { network } = hre;

  if (!tokenType || !newUri) {
    throw new Error('Please provide tokenType and newUri parameters');
  }

  console.log(
    `Updating URI for ${tokenType} token on network ${network.name}...`,
  );
  console.log(`New URI will be: ${newUri}`);

  const contractAddress = getDeployedDATContractAddress(
    network.name,
    tokenType,
  );
  const contract = await ethers.getContractAt(
    'BaseDataAccess',
    contractAddress,
  );

  console.log(`Found ${tokenType} contract at address: ${contractAddress}`);

  console.log('Sending transaction to update URI...');
  const tx = await contract.setURI(newUri);
  await tx.wait();

  console.log('Transaction confirmed. Verifying new URI...');
  const updatedUri = await contract.uri(0);

  if (updatedUri !== newUri) {
    throw new Error(
      `Verification failed! Expected URI ${newUri} but found ${updatedUri}`,
    );
  }

  console.log(
    `Successfully updated and verified URI for ${tokenType} contract.`,
  );
  console.log(`New URI is: ${updatedUri}`);
}

// Main execution pattern to use async/await and handle errors properly
if (require.main === module) {
  const args = process.argv.slice(2);
  if (args.length < 2) {
    console.error('Usage: npx hardhat run updateUri.ts <tokenType> <newUri>');
    process.exit(1);
  }
  const [tokenType, newUri] = args;

  // Import hardhat runtime environment
  import('hardhat').then((hardhat) => {
    const hre = hardhat;

    updateUri(tokenType, newUri, hre)
      .then(() => process.exit(0))
      .catch((error) => {
        console.error(error);
        process.exit(1);
      });
  });
}

exports.updateUri = updateUri;
