import { ethers } from 'hardhat';
import { getSelectors, FacetCutAction } from './libraries/diamond';
import {
  DiamondFullInfo,
  FacetInfo,
  readLastDeployedDiamondInfo,
  saveDiamondInfo,
} from '../contract-helpers';

import { deployBourseLibs } from './deployBourse';

export async function upgradeDiamond(
  diamondName: string,
  facetName: string,
  props: { saveInfo: boolean } = {
    saveInfo: true,
  },
): Promise<void> {
  console.log(`
-----------------------------------------------------
- Upgrading diamond ${diamondName} facet ${facetName}...
- Network: ${process.env.HARDHAT_NETWORK}
-----------------------------------------------------
  `);

  const previousDiamondFullInfo: DiamondFullInfo = readLastDeployedDiamondInfo(
    process.env.HARDHAT_NETWORK,
    diamondName,
  );

  if (!previousDiamondFullInfo || !previousDiamondFullInfo.diamondAddress) {
    throw new Error(
      'Diamond address not found. Make sure the diamond is deployed.',
    );
  }

  const diamondAddress = previousDiamondFullInfo.diamondAddress;
  console.log(`----> Connected to Diamond at address: ${diamondAddress}`);

  // Deploy the new facet
  let Facet;
  if (facetName === 'BourseFacet') {
    // Bourse Facet links libraries.
    const bourseLibAddresses = await deployBourseLibs();
    Facet = await ethers.getContractFactory(facetName, {
      libraries: {
        AVLTreeLib: bourseLibAddresses.avlTreeLibAddress,
        TokenUtilsLib: bourseLibAddresses.tokenUtilsLibAddress,
        BourseLib: bourseLibAddresses.bourseLibAddress,
      },
    });
  } else {
    // Others (Governance, TradingManagement do not link libraries)
    Facet = await ethers.getContractFactory(facetName);
  }

  const facet = await Facet.deploy();
  await facet.deployed();
  console.log(`New Facet Contract ${facetName} deployed: ${facet.address}`);

  // Get selectors for the new facet
  let newSelectors = getSelectors(facet);

  // For Bourse selector for 'supportsInterface(bytes4)' is multiply defined, therefore remove it.
  if (facetName === 'BourseFacet') {
    newSelectors = newSelectors.filter((str) => str !== '0x01ffc9a7');
  }

  // Get the loupe facet to query the diamond's current state
  const loupeFacet = await ethers.getContractAt(
    'IDiamondLoupe',
    diamondAddress,
  );

  const oldFacetInfo = previousDiamondFullInfo.facets[facetName];
  let selectorsToRemove: string[] = [];

  if (oldFacetInfo && oldFacetInfo.length > 0) {
    const oldFacetAddress = oldFacetInfo[oldFacetInfo.length - 1].facetAddress;
    console.log(
      `Querying existing functions for old facet at ${oldFacetAddress}`,
    );
    selectorsToRemove =
      await loupeFacet.facetFunctionSelectors(oldFacetAddress);
  } else {
    console.log(
      `No old facet found for ${facetName}. This will be an Add operation.`,
    );
  }

  // Prepare the diamond cut. We explicitly Add and Remove.
  const diamondCut = [];

  // If there are old selectors, prepare a Remove cut
  if (selectorsToRemove.length > 0) {
    diamondCut.push({
      facetAddress: ethers.constants.AddressZero,
      action: FacetCutAction.Remove,
      functionSelectors: selectorsToRemove,
    });
  }

  // Prepare an Add cut for the new selectors
  diamondCut.push({
    facetAddress: facet.address,
    action: FacetCutAction.Add,
    functionSelectors: newSelectors,
  });

  // Get the diamondCutFacet from the diamond
  const diamondCutFacet = await ethers.getContractAt(
    'IDiamondCut',
    diamondAddress,
  );

  console.log(
    '---> Upgrading diamond by calling diamondCut -------------------------',
  );

  // Perform the diamond cut
  const tx = await diamondCutFacet.diamondCut(
    diamondCut,
    ethers.constants.AddressZero,
    '0x',
  );
  console.log('Diamond cut tx: ', tx.hash);
  const receipt = await tx.wait();
  if (!receipt.status) {
    throw new Error(`Diamond upgrade failed: ${tx.hash}`);
  }
  console.log(`Completed diamond cut, upgraded facet ${facetName}`);

  // Update the diamond info with the new facet address
  const updatedFacetInfo: FacetInfo = {
    publishData: new Date().toISOString(),
    facetName: facetName,
    facetAddress: facet.address,
  };

  if (!previousDiamondFullInfo.facets[facetName]) {
    previousDiamondFullInfo.facets[facetName] = [];
  }
  previousDiamondFullInfo.facets[facetName].push(updatedFacetInfo);
  console.log('Updated Diamond info for saving:', previousDiamondFullInfo);

  if (props.saveInfo) {
    saveDiamondInfo(process.env.HARDHAT_NETWORK, previousDiamondFullInfo);
  } else {
    console.log('Updated Diamond info:', previousDiamondFullInfo);
  }
}

// Main execution pattern to use async/await and handle errors properly
if (require.main === module) {
  const args = process.argv.slice(2);
  if (args.length < 1) {
    console.error('Usage: npx hardhat run upgrade-facet.ts <FacetName>');
    process.exit(1);
  }
  const [diamondName, facetName] = args;
  upgradeDiamond(diamondName, facetName)
    .then(() => process.exit(0))
    .catch((error) => {
      console.error(error);
      process.exit(1);
    });
}

exports.upgradeFacet = upgradeDiamond;
