import { upgrades, ethers } from 'hardhat';
import '@openzeppelin/hardhat-upgrades';
import * as dotenv from 'dotenv';

const envFile = process.env.NODE_ENV ? `.env.${process.env.NODE_ENV}` : '.env';
console.log(`Loading env file: ${envFile}`);
dotenv.config({ path: envFile, override: true });

export async function upgradeTransparentProxyContract({
  contractName,
  proxyContractAddress,
}: {
  contractName: string;
  proxyContractAddress: string;
}): Promise<void> {
  const ContractFactory = await ethers.getContractFactory(contractName);
  if (!ContractFactory) {
    throw new Error(`Contract ${contractName} not found`);
  }
  try {
    await upgrades.upgradeProxy(proxyContractAddress, ContractFactory);
    console.log(
      `Upgraded ${contractName} at address ${proxyContractAddress} successfully`,
    );
  } catch (error: any) {
    if (error.message.includes('is not registered')) {
      console.warn(
        `Proxy at ${proxyContractAddress} not registered. Attempting to force import...`,
      );
      // Use forceImport to register the existing proxy
      await upgrades.forceImport(proxyContractAddress, ContractFactory);

      console.log(
        `Proxy at ${proxyContractAddress} successfully force imported.`,
      );

      // Now that it's imported, try the upgrade again
      console.log('Attempting upgrade again after force import...');
      await upgrades.upgradeProxy(proxyContractAddress, ContractFactory);
      console.log(
        `Upgraded ${contractName} at address ${proxyContractAddress} successfully after force import.`,
      );
    } else {
      console.error('An error occurred during upgrade:');
      throw error; // Re-throw if it's not a registration error or force is not enabled
    }
  }
}
