import { MongoClient } from 'mongodb';
import { HttpException, HttpStatus, Injectable } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';

const collectionName = 'offerings';

@Injectable()
export class PatInternalOfferingRepository {

  private readonly client: MongoClient;
  private readonly dbName: string;
  private readonly url: string;

  constructor(private readonly configService: ConfigService) {
    const env = this.configService.get<string>('APP_ENV');
    this.dbName = this.configService.get<string>('DB_NAME') || 'default_db';

    if (env === 'PROD') {
      const password = this.configService.get<string>('MONGO_INITDB_ROOT_PASSWORD');
      this.url = `mongodb://root:${password}@${this.dbName}:27017`;
    } else {
      const devUrl = this.configService.get<string>('MONGO_DEV_URL');
      if (!devUrl) throw new Error('MONGO_DEV_URL not defined in .env');
      this.url = devUrl;
    }

    this.client = new MongoClient(this.url);
    console.log(`MongoDB URL: ${this.url}`);
  }

  private async connectToDb() {
    try {
      await this.client.connect();
      console.log(`Connected to MongoDB: DB=${this.dbName}, Collection=${collectionName}`);
      return this.client.db(this.dbName).collection(collectionName);
    } catch (err) {
      console.error('Failed to connect to MongoDB', err);
      throw new HttpException(
        { status: HttpStatus.INTERNAL_SERVER_ERROR, error: 'Failed to connect to MongoDB' },
        HttpStatus.INTERNAL_SERVER_ERROR,
      );
    }
  }

  private async closeConnection() {
    await this.client.close();
  }

  async updateOfferingIdByCid(cid: string, newOfferingId: string): Promise<void> {
    const collection = await this.connectToDb();
    try {
      const result = await collection.updateOne(
        { cid: cid },
        { $set: { offering_id: newOfferingId } },
      );

      if (result.matchedCount === 0) {
        throw new HttpException(
          { status: HttpStatus.NOT_FOUND, error: `No offering found with cid: ${cid}` },
          HttpStatus.NOT_FOUND,
        );
      }

      console.log(`Offering with cid ${cid} updated with new offering_id ${newOfferingId}`);
    } catch (error) {
      if (error instanceof HttpException) {
        throw error;
      }
      console.error('Error updating offering by cid:', error);
      throw new HttpException(
        { status: HttpStatus.INTERNAL_SERVER_ERROR, error: 'Failed to update offering by cid' },
        HttpStatus.INTERNAL_SERVER_ERROR,
      );
    } finally {
      await this.closeConnection();
    }
  }

}
